import * as ec2 from '@aws-cdk/aws-ec2';
import * as route53 from '@aws-cdk/aws-route53resolver';
import * as cdk from '@aws-cdk/core';

const FORWARD_DOMAINS = ['justin.tv', 'us-west2.twitch.tv'];

interface TwitchDnsResolverStackProps extends cdk.StackProps {
  twitchDnsServers: string[];
  vpc: ec2.IVpc;
  vpcPrivSubnets: ec2.ISubnet[];
}

// https://wiki.twitch.com/display/PS/How+to+Configure+a+VPC+Resolver+for+Internal+Twitch+DNS
export class TwitchDnsResolverStack extends cdk.Stack {
  constructor(scope: cdk.Construct, name: string, props: TwitchDnsResolverStackProps) {
    super(scope, name, props);

    // The security group must allow outbound TCP and UDP traffic on port 53.
    const securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', {
      vpc: props.vpc,
      allowAllOutbound: false,
      description: 'Allow outbound access to port 53',
    });
    securityGroup.connections.allowTo(ec2.Peer.anyIpv4(), ec2.Port.tcp(53));
    securityGroup.connections.allowTo(ec2.Peer.anyIpv4(), ec2.Port.udp(53));
    securityGroup.connections.allowTo(ec2.Peer.anyIpv6(), ec2.Port.tcp(53));
    securityGroup.connections.allowTo(ec2.Peer.anyIpv6(), ec2.Port.udp(53));

    const endpoint = new route53.CfnResolverEndpoint(this, 'VpcResolverEndpoint', {
      direction: 'OUTBOUND',
      ipAddresses: props.vpcPrivSubnets.map((s) => ({subnetId: s.subnetId})),
      securityGroupIds: [securityGroup.securityGroupId],
    });

    FORWARD_DOMAINS.forEach((domain) => {
      const ruleName = fwdRuleName(domain);
      const rule = new route53.CfnResolverRule(this, `${ruleName}Rule`, {
        domainName: domain,
        ruleType: 'FORWARD',
        name: ruleName,
        resolverEndpointId: endpoint.attrResolverEndpointId,
        targetIps: props.twitchDnsServers.map((dnsServer) => ({ip: dnsServer})),
      })

      new route53.CfnResolverRuleAssociation(this, `${ruleName}RuleAssociation`, {
        resolverRuleId: rule.attrResolverRuleId,
        vpcId: props.vpc.vpcId,
      });
    })

   const dhcpOptions = new ec2.CfnDHCPOptions(this, 'DhcpOptions', {
     domainName: props.env ? `${props.env.region}.compute.internal` : undefined,
     domainNameServers: ['AmazonProvidedDNS'],
     ntpServers: ['169.254.169.123'],
   });

   new ec2.CfnVPCDHCPOptionsAssociation(this, 'VpcDhcpOptionsAssociation', {
     dhcpOptionsId: dhcpOptions.ref,
     vpcId: props.vpc.vpcId,
   })
  }
}

function fwdRuleName(domain: string): string {
  let ruleName = 'Fwd';
  domain.split('.').forEach((s) => {
    ruleName += s.substr(0, 1).toUpperCase() + s.substr(1);
  })
  return ruleName;
}
