package piper

import (
	"context"
	"fmt"
	"net/url"

	"strconv"

	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/insights/piper-service/models"
)

//go:generate mockery -name Client
type Client interface {
	GetGamesReports(ctx context.Context, userid string, domain string, opts *twitchclient.ReqOpts) (*models.GameReportsResp, error)
	GetGamesInstances(ctx context.Context, userid string, domain string, key string, reportName string, opts *twitchclient.ReqOpts) (*models.GameInstancesResp, error)
	GetGamesReportTypes(ctx context.Context, userid string, domain string, key string, opts *twitchclient.ReqOpts) (*models.GameReportTypeResp, error)
	GetAllReportDownloadURLs(ctx context.Context, prop *models.HelixInputProp, opts *twitchclient.ReqOpts) (*models.HelixReportsResp, error)
	GetInstances(ctx context.Context, userid string, domain string, key string, reportName string, opts *twitchclient.ReqOpts) (*models.InstancesResp, error)
	GetReportURL(ctx context.Context, userid string, domain string, key string, reportName string, instance string, orgID string, opts *twitchclient.ReqOpts) (string, error)
	GetReportTypes(ctx context.Context, userid string, domain string, key string, opts *twitchclient.ReqOpts) (*models.ReportTypeResp, error)
	CreateImageUploadRequest(ctx context.Context, userid, gameID string, opts *twitchclient.ReqOpts) (*models.BoxArtUploadResp, error)
	GetGamesOnePagerData(ctx context.Context, userID, gameID string, quarter, year int, addImage bool, opts *twitchclient.ReqOpts) (models.GameOnePagerDataResponse, error)
	GetGamesOnePagerReportURL(ctx context.Context, userID, gameID string, quarter, year int, opts *twitchclient.ReqOpts) (string, error)
	GetReports(ctx context.Context, userid, domain string, opts *twitchclient.ReqOpts) (*models.ReportsResp, error)
	GetReportUrlLatest(ctx context.Context, userid, domain, key, reportName string, opts *twitchclient.ReqOpts) (string, error)
	GetQuarterStartRange(ctx context.Context, userID, gameID string, isDefault bool, opts *twitchclient.ReqOpts) (models.QuarterStartRange, error)
}

const helixEmptyStr = "\"\""

type client struct {
	twitchclient.Client
}

func NewClient(conf twitchclient.ClientConf) (Client, error) {
	twitchClient, err := twitchclient.NewClient(conf)
	return &client{twitchClient}, err
}

func (c *client) GetGamesReports(ctx context.Context, userid string, domain string, opts *twitchclient.ReqOpts) (*models.GameReportsResp, error) {
	endpoint := fmt.Sprintf("/report/%s/%s", url.PathEscape(domain), url.PathEscape(userid))
	req, err := c.NewRequest("GET", endpoint, nil)
	if err != nil {
		return nil, err
	}

	var resp models.GameReportsResp
	_, err = c.DoJSON(ctx, &resp, req, *opts)
	return &resp, err
}

func (c *client) GetGamesInstances(ctx context.Context, userid string, domain string, key string, reportName string, opts *twitchclient.ReqOpts) (*models.GameInstancesResp, error) {
	endpoint := fmt.Sprintf("/report/%s/%s/%s/%s", url.PathEscape(domain), url.PathEscape(userid), url.PathEscape(key), url.PathEscape(reportName))
	req, err := c.NewRequest("GET", endpoint, nil)
	if err != nil {
		return nil, err
	}

	var resp models.GameInstancesResp
	_, err = c.DoJSON(ctx, &resp, req, *opts)
	return &resp, err
}

func (c *client) GetInstances(ctx context.Context, userid string, domain string, key string, reportName string, opts *twitchclient.ReqOpts) (*models.InstancesResp, error) {
	endpoint := fmt.Sprintf("/report/%s/%s/%s/%s", url.PathEscape(domain), url.PathEscape(userid), url.PathEscape(key), url.PathEscape(reportName))
	req, err := c.NewRequest("GET", endpoint, nil)
	if err != nil {
		return nil, err
	}

	var resp models.InstancesResp
	_, err = c.DoJSON(ctx, &resp, req, *opts)
	return &resp, err
}

func (c *client) GetReportURL(ctx context.Context, userid string, domain string, key string, reportName string, instance string, orgID string, opts *twitchclient.ReqOpts) (string, error) {
	// the org_id query param is meant to be used with Drops reports, specifically authorizing with RBAC
	// for other reports, this can be an empty string
	endpoint := fmt.Sprintf("/report/%s/%s/%s/%s/%s?org_id=%s", url.PathEscape(domain), url.PathEscape(userid), url.PathEscape(key), url.PathEscape(reportName), url.PathEscape(instance), url.PathEscape(orgID))
	req, err := c.NewRequest("GET", endpoint, nil)
	if err != nil {
		return "", err
	}

	var resp interface{}
	_, err = c.DoJSON(ctx, &resp, req, *opts)

	url := ""
	if err == nil {
		respMap, ok := resp.(map[string]interface{})
		if ok {
			urlObj, ok := respMap["url"]
			if ok {
				url, _ = urlObj.(string)
			}
		}
	}

	return url, err
}

func (c *client) GetGamesReportTypes(ctx context.Context, userid string, domain string, key string, opts *twitchclient.ReqOpts) (*models.GameReportTypeResp, error) {
	endpoint := fmt.Sprintf("/report/%s/%s/%s", url.PathEscape(domain), url.PathEscape(userid), url.PathEscape(key))
	req, err := c.NewRequest("GET", endpoint, nil)
	if err != nil {
		return nil, err
	}

	var resp models.GameReportTypeResp
	_, err = c.DoJSON(ctx, &resp, req, *opts)
	return &resp, err
}

func (c *client) GetReportTypes(ctx context.Context, userid string, domain string, key string, opts *twitchclient.ReqOpts) (*models.ReportTypeResp, error) {
	endpoint := fmt.Sprintf("/report/%s/%s/%s", url.PathEscape(domain), url.PathEscape(userid), url.PathEscape(key))
	req, err := c.NewRequest("GET", endpoint, nil)
	if err != nil {
		return nil, err
	}

	var resp models.ReportTypeResp
	_, err = c.DoJSON(ctx, &resp, req, *opts)
	return &resp, err
}

var domainIDMap = map[string]string{
	"games":      "game_id",
	"extensions": "extension_id",
	"mods":       "mod_id",
	"channels":   "channel_id",
}

//func (c *client) GetAllReportDownloadURLs(ctx context.Context, userid string, domain string, key string, reportType string, pagination *models.Pagination, startDate string, endDate string, opts *twitchclient.ReqOpts) (*models.HelixReportsResp, error) {
func (c *client) GetAllReportDownloadURLs(ctx context.Context, prop *models.HelixInputProp, opts *twitchclient.ReqOpts) (*models.HelixReportsResp, error) {
	endpoint := fmt.Sprintf("/helix/report/%s/%s", url.PathEscape(prop.Domain), url.PathEscape(prop.UserID))
	queryParams := url.Values{}

	idField, ok := domainIDMap[prop.Domain]
	if prop.Key != "" && prop.Key != helixEmptyStr && ok {
		queryParams.Set(idField, prop.Key)
	}

	if prop.ReportType != "" && prop.ReportType != helixEmptyStr {
		queryParams.Set("type", prop.ReportType)
	}

	if prop.StartDate != "" && prop.ReportType != helixEmptyStr {
		queryParams.Set("start_date", prop.StartDate)
	}

	if prop.EndDate != "" && prop.ReportType != helixEmptyStr {
		queryParams.Set("end_date", prop.EndDate)
	}

	if &prop.Pagination != nil {
		queryParams.Set("limit", strconv.Itoa(prop.Pagination.Limit))
		queryParams.Set("offset", strconv.Itoa(prop.Pagination.Offset))
	}

	requestURL := url.URL{Path: endpoint, RawQuery: queryParams.Encode()}

	resp := models.HelixReportsResp{Data: []models.HelixReport{}}
	req, err := c.NewRequest("GET", requestURL.String(), nil)
	if err != nil {
		return &resp, err
	}

	_, err = c.DoJSON(ctx, &resp, req, *opts)
	return &resp, err
}

func (c *client) CreateImageUploadRequest(ctx context.Context, userid, key string, opts *twitchclient.ReqOpts) (*models.BoxArtUploadResp, error) {
	endpoint := fmt.Sprintf("/boxart/upload/%s/%s", url.PathEscape(userid), url.PathEscape(key))
	req, err := c.NewRequest("GET", endpoint, nil)
	if err != nil {
		return nil, err
	}

	var resp models.BoxArtUploadResp
	_, err = c.DoJSON(ctx, &resp, req, *opts)
	return &resp, err
}

func (c *client) GetGamesOnePagerData(ctx context.Context, userID, gameID string, quarter, year int, addImage bool, opts *twitchclient.ReqOpts) (models.GameOnePagerDataResponse, error) {
	endpoint := fmt.Sprintf("/onepager/games/%s/%s", url.PathEscape(userID), url.PathEscape(gameID))
	queryParams := url.Values{}
	queryParams.Set("quarter", strconv.Itoa(quarter))
	queryParams.Set("year", strconv.Itoa(year))
	queryParams.Set("add_user_image", strconv.FormatBool(addImage))

	requestURL := url.URL{Path: endpoint, RawQuery: queryParams.Encode()}

	req, err := c.NewRequest("GET", requestURL.String(), nil)
	if err != nil {
		return models.GameOnePagerDataResponse{}, err
	}

	var resp models.GameOnePagerDataResponse
	_, err = c.DoJSON(ctx, &resp, req, *opts)
	return resp, err
}

func (c *client) GetGamesOnePagerReportURL(ctx context.Context, userID, gameID string, quarter, year int, opts *twitchclient.ReqOpts) (string, error) {
	endpoint := fmt.Sprintf("/onepager/games/%s/%s/download", url.PathEscape(userID), url.PathEscape(gameID))
	queryParams := url.Values{}
	queryParams.Set("quarter", strconv.Itoa(quarter))
	queryParams.Set("year", strconv.Itoa(year))

	requestURL := url.URL{Path: endpoint, RawQuery: queryParams.Encode()}

	req, err := c.NewRequest("GET", requestURL.String(), nil)
	if err != nil {
		return "", err
	}

	var resp interface{}
	_, err = c.DoJSON(ctx, &resp, req, *opts)
	url := ""
	if err == nil {
		respMap, ok := resp.(map[string]interface{})
		if ok {
			urlObj, ok := respMap["url"]
			if ok {
				url, _ = urlObj.(string)
			}
		}
	}

	return url, err
}

func (c *client) GetReports(ctx context.Context, userid, domain string, opts *twitchclient.ReqOpts) (*models.ReportsResp, error) {
	endpoint := fmt.Sprintf("/report/%s/%s", url.PathEscape(domain), url.PathEscape(userid))
	req, err := c.NewRequest("GET", endpoint, nil)
	if err != nil {
		return nil, err
	}

	var resp models.ReportsResp
	_, err = c.DoJSON(ctx, &resp, req, *opts)
	return &resp, err
}

func (c *client) GetReportUrlLatest(ctx context.Context, userid, domain, key, reportName string, opts *twitchclient.ReqOpts) (string, error) {
	endpoint := fmt.Sprintf("/report/%s/%s/%s/%s/latest", url.PathEscape(domain), url.PathEscape(userid), url.PathEscape(key), url.PathEscape(reportName))
	req, err := c.NewRequest("GET", endpoint, nil)
	if err != nil {
		return "", err
	}

	var resp interface{}
	_, err = c.DoJSON(ctx, &resp, req, *opts)

	url := ""
	if err == nil {
		respMap, ok := resp.(map[string]interface{})
		if ok {
			urlObj, ok := respMap["url"]
			if ok {
				url, _ = urlObj.(string)
			}
		}
	}

	return url, err
}

func (c *client) GetQuarterStartRange(ctx context.Context, userID, gameID string, isDefault bool, opts *twitchclient.ReqOpts) (models.QuarterStartRange, error) {
	endpoint := fmt.Sprintf("/onepager/games/%s/%s/quarterstartrange?default=%s", url.PathEscape(userID), url.PathEscape(gameID), strconv.FormatBool(isDefault))
	req, err := c.NewRequest("GET", endpoint, nil)
	if err != nil {
		return models.QuarterStartRange{}, err
	}

	var resp models.QuarterStartRange
	_, err = c.DoJSON(ctx, &resp, req, *opts)
	return resp, err
}
