package piper_test

import (
	"context"
	"testing"

	"github.com/davecgh/go-spew/spew"

	"code.justin.tv/foundation/twitchclient"
	piper "code.justin.tv/insights/piper-service/client"
	"code.justin.tv/insights/piper-service/models"

	"github.com/stretchr/testify/assert"
	httpmock "gopkg.in/jarcoal/httpmock.v1"
)

func TestGetGamesReports(t *testing.T) {
	httpmock.Activate()
	defer httpmock.DeactivateAndReset()

	httpmock.RegisterResponder("GET", "https://piper-service.prod.internal.twitch.tv/report/games/testuser",
		httpmock.NewStringResponder(200, `{"reports": [{"game_id":"1", "report":"default"}, {"game_id":"5", "report":"monthly"}]}`))

	client, err := piper.NewClient(twitchclient.ClientConf{
		Host:          "https://piper-service.prod.internal.twitch.tv/",
		BaseTransport: httpmock.DefaultTransport,
	})

	assert.Nil(t, err)
	assert.NotNil(t, client)

	gameReports, err := client.GetGamesReports(context.Background(), "testuser", "games", &twitchclient.ReqOpts{})

	assert.Nil(t, err)
	assert.NotNil(t, gameReports)
	assert.Len(t, gameReports.Key, 2)

	assert.NotNil(t, gameReports.Key[0])
	assert.Equal(t, "1", gameReports.Key[0].GameID)
	assert.Equal(t, "default", gameReports.Key[0].ReportName)

	assert.NotNil(t, gameReports.Key[1])
	assert.Equal(t, "5", gameReports.Key[1].GameID)
	assert.Equal(t, "monthly", gameReports.Key[1].ReportName)
}

func TestGetGamesInstances(t *testing.T) {
	httpmock.Activate()
	defer httpmock.DeactivateAndReset()

	httpmock.RegisterResponder("GET", "https://piper-service.prod.internal.twitch.tv/report/games/testuser/1/default",
		httpmock.NewStringResponder(200, `{"report":{"game": "A Great Game", "report": "Default", "instances": ["1234", "5678"]}}`))

	client, err := piper.NewClient(twitchclient.ClientConf{
		Host:          "https://piper-service.prod.internal.twitch.tv/",
		BaseTransport: httpmock.DefaultTransport,
	})

	assert.Nil(t, err)
	assert.NotNil(t, client)

	gameInstances, err := client.GetGamesInstances(context.Background(), "testuser", "games", "1", "default", &twitchclient.ReqOpts{})

	assert.Nil(t, err)
	assert.NotNil(t, gameInstances)
	assert.Equal(t, "A Great Game", gameInstances.Key.GameID)
	assert.Equal(t, "Default", gameInstances.Key.ReportName)
	assert.Len(t, gameInstances.Key.Instances, 2)
	assert.Equal(t, "1234", gameInstances.Key.Instances[0])
	assert.Equal(t, "5678", gameInstances.Key.Instances[1])
}

func TestGetGamesReportTypes(t *testing.T) {
	httpmock.Activate()
	defer httpmock.DeactivateAndReset()

	httpmock.RegisterResponder("GET", "https://piper-service.prod.internal.twitch.tv/report/games/testuser/1234",
		httpmock.NewStringResponder(200, `{"reports":{"game": "1234", "reports": ["default", "overview"]}}`))

	client, err := piper.NewClient(twitchclient.ClientConf{
		Host:          "https://piper-service.prod.internal.twitch.tv/",
		BaseTransport: httpmock.DefaultTransport,
	})

	assert.Nil(t, err)
	assert.NotNil(t, client)

	reportTypes, err := client.GetGamesReportTypes(context.Background(), "testuser", "games", "1234", &twitchclient.ReqOpts{})

	assert.Nil(t, err)
	assert.NotNil(t, reportTypes)
	assert.Equal(t, "1234", reportTypes.Key.GameID)
	assert.Len(t, reportTypes.Key.ReportTypes, 2)
	assert.Equal(t, "default", reportTypes.Key.ReportTypes[0])
	assert.Equal(t, "overview", reportTypes.Key.ReportTypes[1])
}

func TestGetGamesReportTypesFailure(t *testing.T) {
	httpmock.Activate()
	defer httpmock.DeactivateAndReset()

	httpmock.RegisterResponder("GET", "https://piper-service.prod.internal.twitch.tv/report/games/testuser/1234",
		httpmock.NewStringResponder(404, `{"status":404,"message":"Game Name Not Found","error":"Not Found"}`))

	client, err := piper.NewClient(twitchclient.ClientConf{
		Host:          "https://piper-service.prod.internal.twitch.tv/",
		BaseTransport: httpmock.DefaultTransport,
	})
	assert.NoError(t, err)

	reportTypes, err := client.GetGamesReportTypes(context.Background(), "testuser", "games", "1234", &twitchclient.ReqOpts{})

	assert.Equal(t, &models.GameReportTypeResp{Key: models.GameReportType{GameID: "", ReportTypes: []string(nil)}}, reportTypes)
	assert.Error(t, err)
}

func TestGetInstances(t *testing.T) {
	httpmock.Activate()
	defer httpmock.DeactivateAndReset()

	httpmock.RegisterResponder("GET", "https://piper-service.prod.internal.twitch.tv/report/extensions/testuser/1/default",
		httpmock.NewStringResponder(200, `{"report_instance":{"id": "j5a5o14cxb8sgwqdr122c16vol8ai9", "report_type": "Default", "instances": ["1234", "5678"]}}`))

	client, err := piper.NewClient(twitchclient.ClientConf{
		Host:          "https://piper-service.prod.internal.twitch.tv/",
		BaseTransport: httpmock.DefaultTransport,
	})

	assert.Nil(t, err)
	assert.NotNil(t, client)

	instances, err := client.GetInstances(context.Background(), "testuser", "extensions", "1", "default", &twitchclient.ReqOpts{})

	assert.Nil(t, err)
	assert.NotNil(t, instances)
	assert.Equal(t, "j5a5o14cxb8sgwqdr122c16vol8ai9", instances.Key.ID)
	assert.Equal(t, "Default", instances.Key.ReportType)
	assert.Len(t, instances.Key.Instances, 2)
	assert.Equal(t, "1234", instances.Key.Instances[0])
	assert.Equal(t, "5678", instances.Key.Instances[1])
}

func TestGetUrl(t *testing.T) {
	httpmock.Activate()
	defer httpmock.DeactivateAndReset()

	httpmock.RegisterResponder("GET", "https://piper-service.prod.internal.twitch.tv/report/games/testuser/1/default/1234",
		httpmock.NewStringResponder(200, `{"url":"https://twitch-piper-reports-dev.s3-us-west-2.amazonaws.com/games/27546/default/1505088000000.csv?X-Amz-Algorithm=AWS4-HMAC-SHA256\u0026X-Amz-Credential=ASIAIRS44FSDB4CXV6TA%2F20171002%2Fus-west-2%2Fs3%2Faws4_request\u0026X-Amz-Date=20171002T214732Z\u0026X-Amz-Expires=900\u0026X-Amz-Security-Token=FQoDYXdzEPb%2F%2F%2F%2F%2F%2F%2F%2F%2F%2FwEaDOXXZ9NFPXv0MrBghiK3A6qosCwJBOeiglpm%2FGKpoHV6j9uXo0Seo3Xv%2FtfHbMNrMOFZpQgN4xYvPDc9lvQRDbO8Qqz3EiFZT7%2BXdQ1GR1rp0auuhy%2BMd2U7ktmoUPIMV0E%2FBSdRTqI4cdpVBjTTO4EOimuukQReNKwowYkqznMNJoCl1n3whML1yPeSKME12tbnAvfNm%2FLIzWiZHLu3t3ojDWw%2BDJWq%2BkKBdngDFGKTaxwyR%2BL4NyrAqBU0THaooFD%2FXEo0VyNrky1OzYjCsUVs%2B1WCuA1NyZRmXK30lKEahz7ae1dHOGII9Gj9dnchVoNBiWH%2FoqacW9w2XbPqFur4RE0QgM3%2BMFOhU94vPHPDT1MqEy5YV6koH%2B%2BdNdWJcCbVtIl0JIriXjY2ctlr09KVF%2B5rw3jsrq%2BqtNscKhLxLfim6ryi1b9hJl6ta%2FNHtDorhdrzubdoXunMRNwpjtGkun4Fn3Efbruyltk%2FzIFhJcDOXRLvye%2F2vubxqy7LZDCE9qPICLzlprnKHprHWdefLnlTv0p%2BRYbz6gUFiX%2BptHktZMbvBiihuezEVdRJI4V2I2jd5Jt7G4c4T6EXSVLj98XfuSIouMLKzgU%3D\u0026X-Amz-SignedHeaders=host\u0026X-Amz-Signature=530f824764fa365cba7fce11386a897709052f3d1600028e3052cd675dcf4758"}`))

	client, err := piper.NewClient(twitchclient.ClientConf{
		Host:          "https://piper-service.prod.internal.twitch.tv/",
		BaseTransport: httpmock.DefaultTransport,
	})

	assert.Nil(t, err)
	assert.NotNil(t, client)

	reportLink, err := client.GetReportURL(context.Background(), "testuser", "games", "1", "default", "1234", "", &twitchclient.ReqOpts{})

	assert.Nil(t, err)
	assert.Equal(t,
		"https://twitch-piper-reports-dev.s3-us-west-2.amazonaws.com/games/27546/default/1505088000000.csv?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=ASIAIRS44FSDB4CXV6TA%2F20171002%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20171002T214732Z&X-Amz-Expires=900&X-Amz-Security-Token=FQoDYXdzEPb%2F%2F%2F%2F%2F%2F%2F%2F%2F%2FwEaDOXXZ9NFPXv0MrBghiK3A6qosCwJBOeiglpm%2FGKpoHV6j9uXo0Seo3Xv%2FtfHbMNrMOFZpQgN4xYvPDc9lvQRDbO8Qqz3EiFZT7%2BXdQ1GR1rp0auuhy%2BMd2U7ktmoUPIMV0E%2FBSdRTqI4cdpVBjTTO4EOimuukQReNKwowYkqznMNJoCl1n3whML1yPeSKME12tbnAvfNm%2FLIzWiZHLu3t3ojDWw%2BDJWq%2BkKBdngDFGKTaxwyR%2BL4NyrAqBU0THaooFD%2FXEo0VyNrky1OzYjCsUVs%2B1WCuA1NyZRmXK30lKEahz7ae1dHOGII9Gj9dnchVoNBiWH%2FoqacW9w2XbPqFur4RE0QgM3%2BMFOhU94vPHPDT1MqEy5YV6koH%2B%2BdNdWJcCbVtIl0JIriXjY2ctlr09KVF%2B5rw3jsrq%2BqtNscKhLxLfim6ryi1b9hJl6ta%2FNHtDorhdrzubdoXunMRNwpjtGkun4Fn3Efbruyltk%2FzIFhJcDOXRLvye%2F2vubxqy7LZDCE9qPICLzlprnKHprHWdefLnlTv0p%2BRYbz6gUFiX%2BptHktZMbvBiihuezEVdRJI4V2I2jd5Jt7G4c4T6EXSVLj98XfuSIouMLKzgU%3D&X-Amz-SignedHeaders=host&X-Amz-Signature=530f824764fa365cba7fce11386a897709052f3d1600028e3052cd675dcf4758",
		reportLink)
}

func TestGetReportTypes(t *testing.T) {
	httpmock.Activate()
	defer httpmock.DeactivateAndReset()

	httpmock.RegisterResponder("GET", "https://piper-service.prod.internal.twitch.tv/report/extensions/testuser/1234",
		httpmock.NewStringResponder(200, `{"report_type":{"id": "j5a5o14cxb8sgwqdr122c16vol8ai9", "report_types": ["default", "overview"]}}`))

	client, err := piper.NewClient(twitchclient.ClientConf{
		Host:          "https://piper-service.prod.internal.twitch.tv/",
		BaseTransport: httpmock.DefaultTransport,
	})

	assert.Nil(t, err)
	assert.NotNil(t, client)

	reportTypes, err := client.GetReportTypes(context.Background(), "testuser", "extensions", "1234", &twitchclient.ReqOpts{})

	assert.Nil(t, err)
	assert.NotNil(t, reportTypes)
	assert.Equal(t, "j5a5o14cxb8sgwqdr122c16vol8ai9", reportTypes.Key.ID)
	assert.Len(t, reportTypes.Key.ReportTypes, 2)
	assert.Equal(t, "default", reportTypes.Key.ReportTypes[0])
	assert.Equal(t, "overview", reportTypes.Key.ReportTypes[1])
}

func TestGetReportTypesFailure(t *testing.T) {
	httpmock.Activate()
	defer httpmock.DeactivateAndReset()

	httpmock.RegisterResponder("GET", "https://piper-service.prod.internal.twitch.tv/report/extensions/testuser/1234",
		httpmock.NewStringResponder(404, `{"status":404,"message":"Extensions Not Found","error":"Not Found"}`))

	client, err := piper.NewClient(twitchclient.ClientConf{
		Host:          "https://piper-service.prod.internal.twitch.tv/",
		BaseTransport: httpmock.DefaultTransport,
	})
	assert.NoError(t, err)

	reportTypes, err := client.GetReportTypes(context.Background(), "testuser", "extensions", "1234", &twitchclient.ReqOpts{})

	assert.Equal(t, &models.ReportTypeResp{Key: models.ReportType{ID: "", ReportTypes: []string(nil)}}, reportTypes)
	assert.Error(t, err)
}

func TestGetAllURLs(t *testing.T) {
	httpmock.Activate()
	defer httpmock.DeactivateAndReset()

	httpmock.RegisterResponder("GET", "https://piper-service.prod.internal.twitch.tv/helix/report/games/testuser",
		httpmock.NewStringResponder(200,
			`{"data":[
						{"id":"123",
						"type":"overview",
						"url":"https://twitch-piper-reports-dev.s3-us-west-2.amazonaws.com/dynamic/Game1_overview_2018-01-01_2018-02-01.csv?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=ASIAJ4NJJMX7OHHSIL2A%2F20180227%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20180227T202024Z&X-Amz-Expires=60&X-Amz-Security-Token=FQoDYXdzEMX%2F%2F%2F%2F%2F%2F%2F%2F%2F%2FwEaDO6tMLA%2F9SGKudtoJSK3AzUjwQ1Pauvt3wostnfXqmW1KUtPj%2BI0LnaoTbJLRhP4w5oGgYGuz7Ka8ZdnR3bD3zhz%2BtWUXbxSL51lSJG%2F5HUVhI0I%2BvwWP27Z04jS%2FHkNzYtVIJp3TsYPBn1vceE1GRX9yeOpt81qgmkBwwqr6DWWsRN9lTPBTOw0WegMWWQJBozMA%2Blf%2B44CNeGpYAWfoevuiKLyD3W5nUfMwIyKPTJJvPRJwjRhuRO%2BQHkYC7MVWFVTAeXWsv1DmwPs1vfZW1ltMk6rh138ElLaX0S1dscrgm3VCs%2FvJd42hLtZzqjVTaz2Up8BVK58jcQWRs448GZ318H4KYwF1FJ4HqyCl%2FA2DjnjW5qMI8cugluW6EmyEaZQgV8fY0ULNHgyomaGTro96O9A0wAgINHYOZCpx8KhEushNYTvB1ANwWXSzIWrZDVeLGM%2FS3MjC4td3mg1PnCheKzqWH%2FbrciRFynnBku4AKiE5Pe2NGJKt1oz80yY4ULSAAOZtBMNLr3c3y5cAppjVwkdzuIogNIR%2B1bHkZMuwrfTkdyI5tBCu22k%2BKwLuoZdSI4sMEBgwkxKnpOZUILvRXZjOfEo7uvW1AU%3D&X-Amz-SignedHeaders=host&response-content-disposition=attachment%3Bfilename%3DThe%20Game%20Awards-overview-2018-02-14.csv&X-Amz-Signature=32fd4fa5968ede2484855fd3537c443a761c5cb9fc911e85ba67ad9e08a3bf76",
						"start_date": "2018-01-01",
						"end_date": "2018-02-01"}
					],
					"pagination": null
				}`))

	client, err := piper.NewClient(twitchclient.ClientConf{
		Host:          "https://piper-service.prod.internal.twitch.tv/",
		BaseTransport: httpmock.DefaultTransport,
	})
	assert.NoError(t, err)

	prop := &models.HelixInputProp{
		UserID:     "testuser",
		Domain:     "games",
		Key:        "",
		ReportType: "",
		StartDate:  "",
		EndDate:    "",
		Pagination: &models.Pagination{
			Limit:  100,
			Offset: 1,
		},
	}

	resp, err := client.GetAllReportDownloadURLs(context.Background(), prop, &twitchclient.ReqOpts{})

	assert.Equal(t, &models.HelixReportsResp{
		Data: []models.HelixReport{
			{
				ID:        "123",
				Type:      "overview",
				URL:       "https://twitch-piper-reports-dev.s3-us-west-2.amazonaws.com/dynamic/Game1_overview_2018-01-01_2018-02-01.csv?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=ASIAJ4NJJMX7OHHSIL2A%2F20180227%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20180227T202024Z&X-Amz-Expires=60&X-Amz-Security-Token=FQoDYXdzEMX%2F%2F%2F%2F%2F%2F%2F%2F%2F%2FwEaDO6tMLA%2F9SGKudtoJSK3AzUjwQ1Pauvt3wostnfXqmW1KUtPj%2BI0LnaoTbJLRhP4w5oGgYGuz7Ka8ZdnR3bD3zhz%2BtWUXbxSL51lSJG%2F5HUVhI0I%2BvwWP27Z04jS%2FHkNzYtVIJp3TsYPBn1vceE1GRX9yeOpt81qgmkBwwqr6DWWsRN9lTPBTOw0WegMWWQJBozMA%2Blf%2B44CNeGpYAWfoevuiKLyD3W5nUfMwIyKPTJJvPRJwjRhuRO%2BQHkYC7MVWFVTAeXWsv1DmwPs1vfZW1ltMk6rh138ElLaX0S1dscrgm3VCs%2FvJd42hLtZzqjVTaz2Up8BVK58jcQWRs448GZ318H4KYwF1FJ4HqyCl%2FA2DjnjW5qMI8cugluW6EmyEaZQgV8fY0ULNHgyomaGTro96O9A0wAgINHYOZCpx8KhEushNYTvB1ANwWXSzIWrZDVeLGM%2FS3MjC4td3mg1PnCheKzqWH%2FbrciRFynnBku4AKiE5Pe2NGJKt1oz80yY4ULSAAOZtBMNLr3c3y5cAppjVwkdzuIogNIR%2B1bHkZMuwrfTkdyI5tBCu22k%2BKwLuoZdSI4sMEBgwkxKnpOZUILvRXZjOfEo7uvW1AU%3D&X-Amz-SignedHeaders=host&response-content-disposition=attachment%3Bfilename%3DThe%20Game%20Awards-overview-2018-02-14.csv&X-Amz-Signature=32fd4fa5968ede2484855fd3537c443a761c5cb9fc911e85ba67ad9e08a3bf76",
				StartDate: "2018-01-01",
				EndDate:   "2018-02-01",
			},
		},
		Pagination: nil,
	}, resp)
	assert.NoError(t, err)
}

func TestGetAllURLsWithGameId(t *testing.T) {
	httpmock.Activate()
	defer httpmock.DeactivateAndReset()

	httpmock.RegisterResponder("GET", "https://piper-service.prod.internal.twitch.tv/helix/report/games/testuser?game_id=312&limit=100&offset=1",
		httpmock.NewStringResponder(200,
			`{"data":[
						{"id":"123",
						"type":"overview",
						"url":"https://twitch-piper-reports-dev.s3-us-west-2.amazonaws.com/dynamic/Game1_overview_2018-01-01_2018-02-01.csv?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=ASIAJ4NJJMX7OHHSIL2A%2F20180227%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20180227T202024Z&X-Amz-Expires=60&X-Amz-Security-Token=FQoDYXdzEMX%2F%2F%2F%2F%2F%2F%2F%2F%2F%2FwEaDO6tMLA%2F9SGKudtoJSK3AzUjwQ1Pauvt3wostnfXqmW1KUtPj%2BI0LnaoTbJLRhP4w5oGgYGuz7Ka8ZdnR3bD3zhz%2BtWUXbxSL51lSJG%2F5HUVhI0I%2BvwWP27Z04jS%2FHkNzYtVIJp3TsYPBn1vceE1GRX9yeOpt81qgmkBwwqr6DWWsRN9lTPBTOw0WegMWWQJBozMA%2Blf%2B44CNeGpYAWfoevuiKLyD3W5nUfMwIyKPTJJvPRJwjRhuRO%2BQHkYC7MVWFVTAeXWsv1DmwPs1vfZW1ltMk6rh138ElLaX0S1dscrgm3VCs%2FvJd42hLtZzqjVTaz2Up8BVK58jcQWRs448GZ318H4KYwF1FJ4HqyCl%2FA2DjnjW5qMI8cugluW6EmyEaZQgV8fY0ULNHgyomaGTro96O9A0wAgINHYOZCpx8KhEushNYTvB1ANwWXSzIWrZDVeLGM%2FS3MjC4td3mg1PnCheKzqWH%2FbrciRFynnBku4AKiE5Pe2NGJKt1oz80yY4ULSAAOZtBMNLr3c3y5cAppjVwkdzuIogNIR%2B1bHkZMuwrfTkdyI5tBCu22k%2BKwLuoZdSI4sMEBgwkxKnpOZUILvRXZjOfEo7uvW1AU%3D&X-Amz-SignedHeaders=host&response-content-disposition=attachment%3Bfilename%3DThe%20Game%20Awards-overview-2018-02-14.csv&X-Amz-Signature=32fd4fa5968ede2484855fd3537c443a761c5cb9fc911e85ba67ad9e08a3bf76",
						"start_date": "2018-01-01",
						"end_date": "2018-02-01"}
					],
					"pagination": null
				}`))

	client, err := piper.NewClient(twitchclient.ClientConf{
		Host:          "https://piper-service.prod.internal.twitch.tv/",
		BaseTransport: httpmock.DefaultTransport,
	})
	assert.NoError(t, err)

	prop := &models.HelixInputProp{
		UserID:     "testuser",
		Domain:     "games",
		Key:        "312",
		ReportType: "",
		StartDate:  "",
		EndDate:    "",
		Pagination: &models.Pagination{
			Limit:  100,
			Offset: 1,
		},
	}

	resp, err := client.GetAllReportDownloadURLs(context.Background(), prop, &twitchclient.ReqOpts{})

	assert.Equal(t, &models.HelixReportsResp{
		Data: []models.HelixReport{
			{
				ID:        "123",
				Type:      "overview",
				URL:       "https://twitch-piper-reports-dev.s3-us-west-2.amazonaws.com/dynamic/Game1_overview_2018-01-01_2018-02-01.csv?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=ASIAJ4NJJMX7OHHSIL2A%2F20180227%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20180227T202024Z&X-Amz-Expires=60&X-Amz-Security-Token=FQoDYXdzEMX%2F%2F%2F%2F%2F%2F%2F%2F%2F%2FwEaDO6tMLA%2F9SGKudtoJSK3AzUjwQ1Pauvt3wostnfXqmW1KUtPj%2BI0LnaoTbJLRhP4w5oGgYGuz7Ka8ZdnR3bD3zhz%2BtWUXbxSL51lSJG%2F5HUVhI0I%2BvwWP27Z04jS%2FHkNzYtVIJp3TsYPBn1vceE1GRX9yeOpt81qgmkBwwqr6DWWsRN9lTPBTOw0WegMWWQJBozMA%2Blf%2B44CNeGpYAWfoevuiKLyD3W5nUfMwIyKPTJJvPRJwjRhuRO%2BQHkYC7MVWFVTAeXWsv1DmwPs1vfZW1ltMk6rh138ElLaX0S1dscrgm3VCs%2FvJd42hLtZzqjVTaz2Up8BVK58jcQWRs448GZ318H4KYwF1FJ4HqyCl%2FA2DjnjW5qMI8cugluW6EmyEaZQgV8fY0ULNHgyomaGTro96O9A0wAgINHYOZCpx8KhEushNYTvB1ANwWXSzIWrZDVeLGM%2FS3MjC4td3mg1PnCheKzqWH%2FbrciRFynnBku4AKiE5Pe2NGJKt1oz80yY4ULSAAOZtBMNLr3c3y5cAppjVwkdzuIogNIR%2B1bHkZMuwrfTkdyI5tBCu22k%2BKwLuoZdSI4sMEBgwkxKnpOZUILvRXZjOfEo7uvW1AU%3D&X-Amz-SignedHeaders=host&response-content-disposition=attachment%3Bfilename%3DThe%20Game%20Awards-overview-2018-02-14.csv&X-Amz-Signature=32fd4fa5968ede2484855fd3537c443a761c5cb9fc911e85ba67ad9e08a3bf76",
				StartDate: "2018-01-01",
				EndDate:   "2018-02-01",
			},
		},
		Pagination: nil,
	}, resp)
	assert.NoError(t, err)
}

func TestGetAllURLsWithEverything(t *testing.T) {
	httpmock.Activate()
	defer httpmock.DeactivateAndReset()

	httpmock.RegisterResponder("GET", "https://piper-service.prod.internal.twitch.tv/helix/report/games/testuser?end_date=2018-02-01T00:00:00Z&game_id=312&limit=2&offset=1&start_date=2018-01-01T00:00:00Z&type=overview",
		httpmock.NewStringResponder(200,
			`{"data":[
						{"id":"123",
						"type":"overview",
						"url":"https://twitch-piper-reports-dev.s3-us-west-2.amazonaws.com/dynamic/Game1_overview_2018-01-01_2018-02-01.csv?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=ASIAJ4NJJMX7OHHSIL2A%2F20180227%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20180227T202024Z&X-Amz-Expires=60&X-Amz-Security-Token=FQoDYXdzEMX%2F%2F%2F%2F%2F%2F%2F%2F%2F%2FwEaDO6tMLA%2F9SGKudtoJSK3AzUjwQ1Pauvt3wostnfXqmW1KUtPj%2BI0LnaoTbJLRhP4w5oGgYGuz7Ka8ZdnR3bD3zhz%2BtWUXbxSL51lSJG%2F5HUVhI0I%2BvwWP27Z04jS%2FHkNzYtVIJp3TsYPBn1vceE1GRX9yeOpt81qgmkBwwqr6DWWsRN9lTPBTOw0WegMWWQJBozMA%2Blf%2B44CNeGpYAWfoevuiKLyD3W5nUfMwIyKPTJJvPRJwjRhuRO%2BQHkYC7MVWFVTAeXWsv1DmwPs1vfZW1ltMk6rh138ElLaX0S1dscrgm3VCs%2FvJd42hLtZzqjVTaz2Up8BVK58jcQWRs448GZ318H4KYwF1FJ4HqyCl%2FA2DjnjW5qMI8cugluW6EmyEaZQgV8fY0ULNHgyomaGTro96O9A0wAgINHYOZCpx8KhEushNYTvB1ANwWXSzIWrZDVeLGM%2FS3MjC4td3mg1PnCheKzqWH%2FbrciRFynnBku4AKiE5Pe2NGJKt1oz80yY4ULSAAOZtBMNLr3c3y5cAppjVwkdzuIogNIR%2B1bHkZMuwrfTkdyI5tBCu22k%2BKwLuoZdSI4sMEBgwkxKnpOZUILvRXZjOfEo7uvW1AU%3D&X-Amz-SignedHeaders=host&response-content-disposition=attachment%3Bfilename%3DThe%20Game%20Awards-overview-2018-02-14.csv&X-Amz-Signature=32fd4fa5968ede2484855fd3537c443a761c5cb9fc911e85ba67ad9e08a3bf76",
						"start_date": "2018-01-01",
						"end_date": "2018-02-01"}
					],
					"pagination": {
						"offset": 3,
       				"limit": 1
					}
				}`))

	client, err := piper.NewClient(twitchclient.ClientConf{
		Host:          "https://piper-service.prod.internal.twitch.tv/",
		BaseTransport: httpmock.DefaultTransport,
	})
	assert.NoError(t, err)

	prop := &models.HelixInputProp{
		UserID:     "testuser",
		Domain:     "games",
		Key:        "312",
		ReportType: "overview",
		StartDate:  "2018-01-01T00:00:00Z",
		EndDate:    "2018-02-01T00:00:00Z",
		Pagination: &models.Pagination{
			Limit:  2,
			Offset: 1,
		},
	}

	resp, err := client.GetAllReportDownloadURLs(context.Background(), prop, &twitchclient.ReqOpts{})

	assert.Equal(t, &models.HelixReportsResp{
		Data: []models.HelixReport{
			{
				ID:        "123",
				Type:      "overview",
				URL:       "https://twitch-piper-reports-dev.s3-us-west-2.amazonaws.com/dynamic/Game1_overview_2018-01-01_2018-02-01.csv?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=ASIAJ4NJJMX7OHHSIL2A%2F20180227%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20180227T202024Z&X-Amz-Expires=60&X-Amz-Security-Token=FQoDYXdzEMX%2F%2F%2F%2F%2F%2F%2F%2F%2F%2FwEaDO6tMLA%2F9SGKudtoJSK3AzUjwQ1Pauvt3wostnfXqmW1KUtPj%2BI0LnaoTbJLRhP4w5oGgYGuz7Ka8ZdnR3bD3zhz%2BtWUXbxSL51lSJG%2F5HUVhI0I%2BvwWP27Z04jS%2FHkNzYtVIJp3TsYPBn1vceE1GRX9yeOpt81qgmkBwwqr6DWWsRN9lTPBTOw0WegMWWQJBozMA%2Blf%2B44CNeGpYAWfoevuiKLyD3W5nUfMwIyKPTJJvPRJwjRhuRO%2BQHkYC7MVWFVTAeXWsv1DmwPs1vfZW1ltMk6rh138ElLaX0S1dscrgm3VCs%2FvJd42hLtZzqjVTaz2Up8BVK58jcQWRs448GZ318H4KYwF1FJ4HqyCl%2FA2DjnjW5qMI8cugluW6EmyEaZQgV8fY0ULNHgyomaGTro96O9A0wAgINHYOZCpx8KhEushNYTvB1ANwWXSzIWrZDVeLGM%2FS3MjC4td3mg1PnCheKzqWH%2FbrciRFynnBku4AKiE5Pe2NGJKt1oz80yY4ULSAAOZtBMNLr3c3y5cAppjVwkdzuIogNIR%2B1bHkZMuwrfTkdyI5tBCu22k%2BKwLuoZdSI4sMEBgwkxKnpOZUILvRXZjOfEo7uvW1AU%3D&X-Amz-SignedHeaders=host&response-content-disposition=attachment%3Bfilename%3DThe%20Game%20Awards-overview-2018-02-14.csv&X-Amz-Signature=32fd4fa5968ede2484855fd3537c443a761c5cb9fc911e85ba67ad9e08a3bf76",
				StartDate: "2018-01-01",
				EndDate:   "2018-02-01",
			},
		},
		Pagination: &models.Pagination{
			Offset: 3,
			Limit:  1,
		},
	},
		resp)
	assert.NoError(t, err)
}

func TestGetAllURLsFailure(t *testing.T) {
	httpmock.Activate()
	defer httpmock.DeactivateAndReset()

	httpmock.RegisterResponder("GET", "https://piper-service.prod.internal.twitch.tv/helix/report/games/testuser",
		httpmock.NewStringResponder(404, `{"status":404,"message":"Game Name Not Found","error":"Not Found"}`))

	client, err := piper.NewClient(twitchclient.ClientConf{
		Host:          "https://piper-service.prod.internal.twitch.tv/",
		BaseTransport: httpmock.DefaultTransport,
	})
	assert.NoError(t, err)

	prop := &models.HelixInputProp{
		UserID:     "testuser",
		Domain:     "games",
		Key:        "",
		ReportType: "",
		StartDate:  "2018-01-01T00:00:00Z",
		EndDate:    "2018-02-01T00:00:00Z",
		Pagination: &models.Pagination{
			Limit:  10,
			Offset: 0,
		},
	}

	resp, err := client.GetAllReportDownloadURLs(context.Background(), prop, &twitchclient.ReqOpts{})

	assert.Equal(t, &models.HelixReportsResp{
		Data: []models.HelixReport{},
	}, resp)
	assert.Error(t, err)
}

func TestGetAllExtensionReportURLs(t *testing.T) {
	httpmock.Activate()
	defer httpmock.DeactivateAndReset()

	httpmock.RegisterResponder("GET", "https://piper-service.prod.internal.twitch.tv/helix/report/extensions/testuser?limit=10&offset=0",
		httpmock.NewStringResponder(200,
			`{"data":[
						{"id":"123",
						"type":"overview",
						"url":"https://twitch-piper-reports-dev.s3-us-west-2.amazonaws.com/dynamic/Extension1_overview_2018-01-01_2018-02-01.csv?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=ASIAJ4NJJMX7OHHSIL2A%2F20180227%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20180227T202024Z&X-Amz-Expires=60&X-Amz-Security-Token=FQoDYXdzEMX%2F%2F%2F%2F%2F%2F%2F%2F%2F%2FwEaDO6tMLA%2F9SGKudtoJSK3AzUjwQ1Pauvt3wostnfXqmW1KUtPj%2BI0LnaoTbJLRhP4w5oGgYGuz7Ka8ZdnR3bD3zhz%2BtWUXbxSL51lSJG%2F5HUVhI0I%2BvwWP27Z04jS%2FHkNzYtVIJp3TsYPBn1vceE1GRX9yeOpt81qgmkBwwqr6DWWsRN9lTPBTOw0WegMWWQJBozMA%2Blf%2B44CNeGpYAWfoevuiKLyD3W5nUfMwIyKPTJJvPRJwjRhuRO%2BQHkYC7MVWFVTAeXWsv1DmwPs1vfZW1ltMk6rh138ElLaX0S1dscrgm3VCs%2FvJd42hLtZzqjVTaz2Up8BVK58jcQWRs448GZ318H4KYwF1FJ4HqyCl%2FA2DjnjW5qMI8cugluW6EmyEaZQgV8fY0ULNHgyomaGTro96O9A0wAgINHYOZCpx8KhEushNYTvB1ANwWXSzIWrZDVeLGM%2FS3MjC4td3mg1PnCheKzqWH%2FbrciRFynnBku4AKiE5Pe2NGJKt1oz80yY4ULSAAOZtBMNLr3c3y5cAppjVwkdzuIogNIR%2B1bHkZMuwrfTkdyI5tBCu22k%2BKwLuoZdSI4sMEBgwkxKnpOZUILvRXZjOfEo7uvW1AU%3D&X-Amz-SignedHeaders=host&response-content-disposition=attachment%3Bfilename%3DThe%20Game%20Awards-overview-2018-02-14.csv&X-Amz-Signature=32fd4fa5968ede2484855fd3537c443a761c5cb9fc911e85ba67ad9e08a3bf76",
						"start_date": "2018-01-01",
						"end_date": "2018-02-01"}
					],
					"pagination": null
				}`))

	client, err := piper.NewClient(twitchclient.ClientConf{
		Host:          "https://piper-service.prod.internal.twitch.tv/",
		BaseTransport: httpmock.DefaultTransport,
	})
	assert.NoError(t, err)

	prop := &models.HelixInputProp{
		UserID:     "testuser",
		Domain:     "extensions",
		Key:        "",
		ReportType: "",
		StartDate:  "",
		EndDate:    "",
		Pagination: &models.Pagination{
			Limit:  10,
			Offset: 0,
		},
	}
	resp, err := client.GetAllReportDownloadURLs(context.Background(), prop, &twitchclient.ReqOpts{})

	assert.Equal(t, &models.HelixReportsResp{
		Data: []models.HelixReport{
			{
				ID:        "123",
				Type:      "overview",
				URL:       "https://twitch-piper-reports-dev.s3-us-west-2.amazonaws.com/dynamic/Extension1_overview_2018-01-01_2018-02-01.csv?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=ASIAJ4NJJMX7OHHSIL2A%2F20180227%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20180227T202024Z&X-Amz-Expires=60&X-Amz-Security-Token=FQoDYXdzEMX%2F%2F%2F%2F%2F%2F%2F%2F%2F%2FwEaDO6tMLA%2F9SGKudtoJSK3AzUjwQ1Pauvt3wostnfXqmW1KUtPj%2BI0LnaoTbJLRhP4w5oGgYGuz7Ka8ZdnR3bD3zhz%2BtWUXbxSL51lSJG%2F5HUVhI0I%2BvwWP27Z04jS%2FHkNzYtVIJp3TsYPBn1vceE1GRX9yeOpt81qgmkBwwqr6DWWsRN9lTPBTOw0WegMWWQJBozMA%2Blf%2B44CNeGpYAWfoevuiKLyD3W5nUfMwIyKPTJJvPRJwjRhuRO%2BQHkYC7MVWFVTAeXWsv1DmwPs1vfZW1ltMk6rh138ElLaX0S1dscrgm3VCs%2FvJd42hLtZzqjVTaz2Up8BVK58jcQWRs448GZ318H4KYwF1FJ4HqyCl%2FA2DjnjW5qMI8cugluW6EmyEaZQgV8fY0ULNHgyomaGTro96O9A0wAgINHYOZCpx8KhEushNYTvB1ANwWXSzIWrZDVeLGM%2FS3MjC4td3mg1PnCheKzqWH%2FbrciRFynnBku4AKiE5Pe2NGJKt1oz80yY4ULSAAOZtBMNLr3c3y5cAppjVwkdzuIogNIR%2B1bHkZMuwrfTkdyI5tBCu22k%2BKwLuoZdSI4sMEBgwkxKnpOZUILvRXZjOfEo7uvW1AU%3D&X-Amz-SignedHeaders=host&response-content-disposition=attachment%3Bfilename%3DThe%20Game%20Awards-overview-2018-02-14.csv&X-Amz-Signature=32fd4fa5968ede2484855fd3537c443a761c5cb9fc911e85ba67ad9e08a3bf76",
				StartDate: "2018-01-01",
				EndDate:   "2018-02-01",
			},
		},
		Pagination: nil,
	}, resp)
	assert.NoError(t, err)
}

func TestGetAllURLsWithExtensionId(t *testing.T) {
	httpmock.Activate()
	defer httpmock.DeactivateAndReset()

	httpmock.RegisterResponder("GET", "https://piper-service.dev.internal.twitch.tv/helix/report/extensions/testuser?extension_id=123&limit=10&offset=0",
		httpmock.NewStringResponder(200,
			`{"data":[
						{"id":"123",
						"type":"overview",
						"url":"https://twitch-piper-reports-dev.s3-us-west-2.amazonaws.com/dynamic/Extension1_overview_2018-01-01_2018-02-01.csv?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=ASIAJ4NJJMX7OHHSIL2A%2F20180227%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20180227T202024Z&X-Amz-Expires=60&X-Amz-Security-Token=FQoDYXdzEMX%2F%2F%2F%2F%2F%2F%2F%2F%2F%2FwEaDO6tMLA%2F9SGKudtoJSK3AzUjwQ1Pauvt3wostnfXqmW1KUtPj%2BI0LnaoTbJLRhP4w5oGgYGuz7Ka8ZdnR3bD3zhz%2BtWUXbxSL51lSJG%2F5HUVhI0I%2BvwWP27Z04jS%2FHkNzYtVIJp3TsYPBn1vceE1GRX9yeOpt81qgmkBwwqr6DWWsRN9lTPBTOw0WegMWWQJBozMA%2Blf%2B44CNeGpYAWfoevuiKLyD3W5nUfMwIyKPTJJvPRJwjRhuRO%2BQHkYC7MVWFVTAeXWsv1DmwPs1vfZW1ltMk6rh138ElLaX0S1dscrgm3VCs%2FvJd42hLtZzqjVTaz2Up8BVK58jcQWRs448GZ318H4KYwF1FJ4HqyCl%2FA2DjnjW5qMI8cugluW6EmyEaZQgV8fY0ULNHgyomaGTro96O9A0wAgINHYOZCpx8KhEushNYTvB1ANwWXSzIWrZDVeLGM%2FS3MjC4td3mg1PnCheKzqWH%2FbrciRFynnBku4AKiE5Pe2NGJKt1oz80yY4ULSAAOZtBMNLr3c3y5cAppjVwkdzuIogNIR%2B1bHkZMuwrfTkdyI5tBCu22k%2BKwLuoZdSI4sMEBgwkxKnpOZUILvRXZjOfEo7uvW1AU%3D&X-Amz-SignedHeaders=host&response-content-disposition=attachment%3Bfilename%3DThe%20Game%20Awards-overview-2018-02-14.csv&X-Amz-Signature=32fd4fa5968ede2484855fd3537c443a761c5cb9fc911e85ba67ad9e08a3bf76",
						"start_date": "2018-01-01",
						"end_date": "2018-02-01"}
					],
					"pagination": null
				}`))

	client, err := piper.NewClient(twitchclient.ClientConf{
		Host:          "https://piper-service.dev.internal.twitch.tv/",
		BaseTransport: httpmock.DefaultTransport,
	})
	assert.NoError(t, err)

	prop := &models.HelixInputProp{
		UserID:     "testuser",
		Domain:     "extensions",
		Key:        "123",
		ReportType: "",
		StartDate:  "",
		EndDate:    "",
		Pagination: &models.Pagination{
			Limit:  10,
			Offset: 0,
		},
	}
	resp, err := client.GetAllReportDownloadURLs(context.Background(), prop, &twitchclient.ReqOpts{})

	assert.Equal(t, &models.HelixReportsResp{
		Data: []models.HelixReport{
			{
				ID:        "123",
				Type:      "overview",
				URL:       "https://twitch-piper-reports-dev.s3-us-west-2.amazonaws.com/dynamic/Extension1_overview_2018-01-01_2018-02-01.csv?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=ASIAJ4NJJMX7OHHSIL2A%2F20180227%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20180227T202024Z&X-Amz-Expires=60&X-Amz-Security-Token=FQoDYXdzEMX%2F%2F%2F%2F%2F%2F%2F%2F%2F%2FwEaDO6tMLA%2F9SGKudtoJSK3AzUjwQ1Pauvt3wostnfXqmW1KUtPj%2BI0LnaoTbJLRhP4w5oGgYGuz7Ka8ZdnR3bD3zhz%2BtWUXbxSL51lSJG%2F5HUVhI0I%2BvwWP27Z04jS%2FHkNzYtVIJp3TsYPBn1vceE1GRX9yeOpt81qgmkBwwqr6DWWsRN9lTPBTOw0WegMWWQJBozMA%2Blf%2B44CNeGpYAWfoevuiKLyD3W5nUfMwIyKPTJJvPRJwjRhuRO%2BQHkYC7MVWFVTAeXWsv1DmwPs1vfZW1ltMk6rh138ElLaX0S1dscrgm3VCs%2FvJd42hLtZzqjVTaz2Up8BVK58jcQWRs448GZ318H4KYwF1FJ4HqyCl%2FA2DjnjW5qMI8cugluW6EmyEaZQgV8fY0ULNHgyomaGTro96O9A0wAgINHYOZCpx8KhEushNYTvB1ANwWXSzIWrZDVeLGM%2FS3MjC4td3mg1PnCheKzqWH%2FbrciRFynnBku4AKiE5Pe2NGJKt1oz80yY4ULSAAOZtBMNLr3c3y5cAppjVwkdzuIogNIR%2B1bHkZMuwrfTkdyI5tBCu22k%2BKwLuoZdSI4sMEBgwkxKnpOZUILvRXZjOfEo7uvW1AU%3D&X-Amz-SignedHeaders=host&response-content-disposition=attachment%3Bfilename%3DThe%20Game%20Awards-overview-2018-02-14.csv&X-Amz-Signature=32fd4fa5968ede2484855fd3537c443a761c5cb9fc911e85ba67ad9e08a3bf76",
				StartDate: "2018-01-01",
				EndDate:   "2018-02-01",
			},
		},
		Pagination: nil,
	}, resp)
	assert.NoError(t, err)
}

func TestGetAllExtensionURLsWithEverything(t *testing.T) {
	httpmock.Activate()
	defer httpmock.DeactivateAndReset()

	httpmock.RegisterResponder("GET", "https://piper-service.prod.internal.twitch.tv/helix/report/extensions/testuser?end_date=2018-02-01T00:00:00Z&extension_id=123&limit=10&offset=0&start_date=2018-01-01T00:00:00Z&type=overview",
		httpmock.NewStringResponder(200,
			`{"data":[
						{"id":"123",
						"type":"overview",
						"url":"https://twitch-piper-reports-dev.s3-us-west-2.amazonaws.com/dynamic/Extension1_overview_2018-01-01_2018-02-01.csv?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=ASIAJ4NJJMX7OHHSIL2A%2F20180227%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20180227T202024Z&X-Amz-Expires=60&X-Amz-Security-Token=FQoDYXdzEMX%2F%2F%2F%2F%2F%2F%2F%2F%2F%2FwEaDO6tMLA%2F9SGKudtoJSK3AzUjwQ1Pauvt3wostnfXqmW1KUtPj%2BI0LnaoTbJLRhP4w5oGgYGuz7Ka8ZdnR3bD3zhz%2BtWUXbxSL51lSJG%2F5HUVhI0I%2BvwWP27Z04jS%2FHkNzYtVIJp3TsYPBn1vceE1GRX9yeOpt81qgmkBwwqr6DWWsRN9lTPBTOw0WegMWWQJBozMA%2Blf%2B44CNeGpYAWfoevuiKLyD3W5nUfMwIyKPTJJvPRJwjRhuRO%2BQHkYC7MVWFVTAeXWsv1DmwPs1vfZW1ltMk6rh138ElLaX0S1dscrgm3VCs%2FvJd42hLtZzqjVTaz2Up8BVK58jcQWRs448GZ318H4KYwF1FJ4HqyCl%2FA2DjnjW5qMI8cugluW6EmyEaZQgV8fY0ULNHgyomaGTro96O9A0wAgINHYOZCpx8KhEushNYTvB1ANwWXSzIWrZDVeLGM%2FS3MjC4td3mg1PnCheKzqWH%2FbrciRFynnBku4AKiE5Pe2NGJKt1oz80yY4ULSAAOZtBMNLr3c3y5cAppjVwkdzuIogNIR%2B1bHkZMuwrfTkdyI5tBCu22k%2BKwLuoZdSI4sMEBgwkxKnpOZUILvRXZjOfEo7uvW1AU%3D&X-Amz-SignedHeaders=host&response-content-disposition=attachment%3Bfilename%3DThe%20Game%20Awards-overview-2018-02-14.csv&X-Amz-Signature=32fd4fa5968ede2484855fd3537c443a761c5cb9fc911e85ba67ad9e08a3bf76",
						"start_date": "2018-01-01",
						"end_date": "2018-02-01"}
					],
					"pagination": null
				}`))

	client, err := piper.NewClient(twitchclient.ClientConf{
		Host:          "https://piper-service.prod.internal.twitch.tv/",
		BaseTransport: httpmock.DefaultTransport,
	})
	assert.NoError(t, err)

	prop := &models.HelixInputProp{
		UserID:     "testuser",
		Domain:     "extensions",
		Key:        "123",
		ReportType: "overview",
		StartDate:  "2018-01-01T00:00:00Z",
		EndDate:    "2018-02-01T00:00:00Z",
		Pagination: &models.Pagination{
			Limit:  10,
			Offset: 0,
		},
	}

	resp, err := client.GetAllReportDownloadURLs(context.Background(), prop, &twitchclient.ReqOpts{})

	assert.Equal(t, &models.HelixReportsResp{
		Data: []models.HelixReport{
			{
				ID:        "123",
				Type:      "overview",
				URL:       "https://twitch-piper-reports-dev.s3-us-west-2.amazonaws.com/dynamic/Extension1_overview_2018-01-01_2018-02-01.csv?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=ASIAJ4NJJMX7OHHSIL2A%2F20180227%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20180227T202024Z&X-Amz-Expires=60&X-Amz-Security-Token=FQoDYXdzEMX%2F%2F%2F%2F%2F%2F%2F%2F%2F%2FwEaDO6tMLA%2F9SGKudtoJSK3AzUjwQ1Pauvt3wostnfXqmW1KUtPj%2BI0LnaoTbJLRhP4w5oGgYGuz7Ka8ZdnR3bD3zhz%2BtWUXbxSL51lSJG%2F5HUVhI0I%2BvwWP27Z04jS%2FHkNzYtVIJp3TsYPBn1vceE1GRX9yeOpt81qgmkBwwqr6DWWsRN9lTPBTOw0WegMWWQJBozMA%2Blf%2B44CNeGpYAWfoevuiKLyD3W5nUfMwIyKPTJJvPRJwjRhuRO%2BQHkYC7MVWFVTAeXWsv1DmwPs1vfZW1ltMk6rh138ElLaX0S1dscrgm3VCs%2FvJd42hLtZzqjVTaz2Up8BVK58jcQWRs448GZ318H4KYwF1FJ4HqyCl%2FA2DjnjW5qMI8cugluW6EmyEaZQgV8fY0ULNHgyomaGTro96O9A0wAgINHYOZCpx8KhEushNYTvB1ANwWXSzIWrZDVeLGM%2FS3MjC4td3mg1PnCheKzqWH%2FbrciRFynnBku4AKiE5Pe2NGJKt1oz80yY4ULSAAOZtBMNLr3c3y5cAppjVwkdzuIogNIR%2B1bHkZMuwrfTkdyI5tBCu22k%2BKwLuoZdSI4sMEBgwkxKnpOZUILvRXZjOfEo7uvW1AU%3D&X-Amz-SignedHeaders=host&response-content-disposition=attachment%3Bfilename%3DThe%20Game%20Awards-overview-2018-02-14.csv&X-Amz-Signature=32fd4fa5968ede2484855fd3537c443a761c5cb9fc911e85ba67ad9e08a3bf76",
				StartDate: "2018-01-01",
				EndDate:   "2018-02-01",
			},
		},
		Pagination: nil,
	}, resp)
	assert.NoError(t, err)
}

func TestGetAllExtensionURLsFailure(t *testing.T) {
	httpmock.Activate()
	defer httpmock.DeactivateAndReset()

	httpmock.RegisterResponder("GET", "https://piper-service.prod.internal.twitch.tv/helix/report/extensions/testuser",
		httpmock.NewStringResponder(404, `{"status":404,"message":"Resource Not Found","error":"Not Found"}`))

	client, err := piper.NewClient(twitchclient.ClientConf{
		Host:          "https://piper-service.prod.internal.twitch.tv/",
		BaseTransport: httpmock.DefaultTransport,
	})
	assert.NoError(t, err)

	prop := &models.HelixInputProp{
		UserID:     "testuser",
		Domain:     "extensions",
		Key:        "",
		ReportType: "",
		StartDate:  "",
		EndDate:    "",
		Pagination: &models.Pagination{
			Limit:  10,
			Offset: 0,
		},
	}

	resp, err := client.GetAllReportDownloadURLs(context.Background(), prop, &twitchclient.ReqOpts{})

	assert.Equal(t, &models.HelixReportsResp{
		Data: []models.HelixReport{},
	}, resp)
	assert.Error(t, err)
}

func TestCreateImageUploadRequest(t *testing.T) {
	httpmock.Activate()
	defer httpmock.DeactivateAndReset()

	httpmock.RegisterResponder("GET", "https://piper-service.prod.internal.twitch.tv/boxart/upload/testuser/12345",
		httpmock.NewStringResponder(200, `{"upload": {"upload_id": "290e26eb-6710-4fe6-a290-546c97b77b10", "url": "https://staging-web-upload-service-ingest.s3-accelerate.amazonaws.com/290e26eb-6710-4fe6-a290-546c97b77b10?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Content-Sha256=UNSIGNED-PAYLOAD&X-Amz-Credential=ASIAJ2H4E2GER5PZZEXA%2F20180530%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20180530T184120Z&X-Amz-Expires=900&X-Amz-Security-Token=FQoDYXdzEPT%2F%2F%2F%2F%2F%2F%2F%2F%2F%2FwEaDM7njjn2qJtEkJPtliK3A%2FAZdZU4ikskSGL7jWp91KQ14b0VBASynm2wNFCsqwJbvdsrLUQh4ukIGrrEYDklzzcKNKi6kDr1bJDmaAYDwVb5DF2KyEOWTS4B6YXRpLAL%2BBX4e7Wp5%2FWQQ%2BHxYgwz8h1gBuGjOmGUHwY12%2BcR%2BW46sX2zJGgFLRZS0IHQLcr%2FmyIEv%2FPoZ3SDpJIOb%2FC%2FMXF7nMXfD8nsMA7xgulqjdWfnv2vRtFQQFPXKjdPjO3CHUso%2BttCClSlQo%2F9jsKtttfdMShH03WOlC8PKq8hkxFWIg39%2FVoblmosPRj7HFlGAb1ffSFjLrJozPQ37BxgCcHPAxov6WEs7JYda2XdBjBVHpk8i5k%2FIVFbddH0GERLX4DdKwZJlqfaJlG3A1LUAESRan0tyxHSuxux2YoMtNrHnJVNSvU%2BBCfFsuCiPUBtFO3hcDoi21t5pRNp0Lu9fT1RhOS6GgbJE%2BoSUt05uqG5yrykshPwE7nY4h2BL3Th7%2BXef3P64%2BurDbDGHRgxMw5j%2BDATjEc2%2FldKkGn%2FaNo%2BzNBQf3VuacJ1WWosfvfjDzZ9ghSXLah%2FyHzL9GTBoCKzE86DiFgo2Ni72AU%3D&X-Amz-SignedHeaders=host&X-Amz-Signature=f816fce152d9acc391c08c21d76c7a2b3611e4416fcff63b01e171f58f01cbf5"}}`))

	client, err := piper.NewClient(twitchclient.ClientConf{
		Host:          "https://piper-service.prod.internal.twitch.tv/",
		BaseTransport: httpmock.DefaultTransport,
	})
	assert.Nil(t, err)
	assert.NotNil(t, client)

	uploadReq, err := client.CreateImageUploadRequest(context.Background(), "testuser", "12345", &twitchclient.ReqOpts{})

	assert.Nil(t, err)
	assert.NotNil(t, uploadReq)
	assert.Equal(t, "290e26eb-6710-4fe6-a290-546c97b77b10", uploadReq.Key.UploadID)
	assert.Equal(t,
		"https://staging-web-upload-service-ingest.s3-accelerate.amazonaws.com/290e26eb-6710-4fe6-a290-546c97b77b10?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Content-Sha256=UNSIGNED-PAYLOAD&X-Amz-Credential=ASIAJ2H4E2GER5PZZEXA%2F20180530%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20180530T184120Z&X-Amz-Expires=900&X-Amz-Security-Token=FQoDYXdzEPT%2F%2F%2F%2F%2F%2F%2F%2F%2F%2FwEaDM7njjn2qJtEkJPtliK3A%2FAZdZU4ikskSGL7jWp91KQ14b0VBASynm2wNFCsqwJbvdsrLUQh4ukIGrrEYDklzzcKNKi6kDr1bJDmaAYDwVb5DF2KyEOWTS4B6YXRpLAL%2BBX4e7Wp5%2FWQQ%2BHxYgwz8h1gBuGjOmGUHwY12%2BcR%2BW46sX2zJGgFLRZS0IHQLcr%2FmyIEv%2FPoZ3SDpJIOb%2FC%2FMXF7nMXfD8nsMA7xgulqjdWfnv2vRtFQQFPXKjdPjO3CHUso%2BttCClSlQo%2F9jsKtttfdMShH03WOlC8PKq8hkxFWIg39%2FVoblmosPRj7HFlGAb1ffSFjLrJozPQ37BxgCcHPAxov6WEs7JYda2XdBjBVHpk8i5k%2FIVFbddH0GERLX4DdKwZJlqfaJlG3A1LUAESRan0tyxHSuxux2YoMtNrHnJVNSvU%2BBCfFsuCiPUBtFO3hcDoi21t5pRNp0Lu9fT1RhOS6GgbJE%2BoSUt05uqG5yrykshPwE7nY4h2BL3Th7%2BXef3P64%2BurDbDGHRgxMw5j%2BDATjEc2%2FldKkGn%2FaNo%2BzNBQf3VuacJ1WWosfvfjDzZ9ghSXLah%2FyHzL9GTBoCKzE86DiFgo2Ni72AU%3D&X-Amz-SignedHeaders=host&X-Amz-Signature=f816fce152d9acc391c08c21d76c7a2b3611e4416fcff63b01e171f58f01cbf5",
		uploadReq.Key.URL)
}

func TestGetAllURLsWithHelixEmptyID(t *testing.T) {
	httpmock.Activate()
	defer httpmock.DeactivateAndReset()

	httpmock.RegisterResponder("GET", "https://piper-service.prod.internal.twitch.tv/helix/report/games/testuser?limit=100&offset=1",
		httpmock.NewStringResponder(200,
			`{"data":[
						{"id":"123",
						"type":"overview",
						"url":"https://twitch-piper-reports-dev.s3-us-west-2.amazonaws.com/dynamic/Game1_overview_2018-01-01_2018-02-01.csv?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=ASIAJ4NJJMX7OHHSIL2A%2F20180227%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20180227T202024Z&X-Amz-Expires=60&X-Amz-Security-Token=FQoDYXdzEMX%2F%2F%2F%2F%2F%2F%2F%2F%2F%2FwEaDO6tMLA%2F9SGKudtoJSK3AzUjwQ1Pauvt3wostnfXqmW1KUtPj%2BI0LnaoTbJLRhP4w5oGgYGuz7Ka8ZdnR3bD3zhz%2BtWUXbxSL51lSJG%2F5HUVhI0I%2BvwWP27Z04jS%2FHkNzYtVIJp3TsYPBn1vceE1GRX9yeOpt81qgmkBwwqr6DWWsRN9lTPBTOw0WegMWWQJBozMA%2Blf%2B44CNeGpYAWfoevuiKLyD3W5nUfMwIyKPTJJvPRJwjRhuRO%2BQHkYC7MVWFVTAeXWsv1DmwPs1vfZW1ltMk6rh138ElLaX0S1dscrgm3VCs%2FvJd42hLtZzqjVTaz2Up8BVK58jcQWRs448GZ318H4KYwF1FJ4HqyCl%2FA2DjnjW5qMI8cugluW6EmyEaZQgV8fY0ULNHgyomaGTro96O9A0wAgINHYOZCpx8KhEushNYTvB1ANwWXSzIWrZDVeLGM%2FS3MjC4td3mg1PnCheKzqWH%2FbrciRFynnBku4AKiE5Pe2NGJKt1oz80yY4ULSAAOZtBMNLr3c3y5cAppjVwkdzuIogNIR%2B1bHkZMuwrfTkdyI5tBCu22k%2BKwLuoZdSI4sMEBgwkxKnpOZUILvRXZjOfEo7uvW1AU%3D&X-Amz-SignedHeaders=host&response-content-disposition=attachment%3Bfilename%3DThe%20Game%20Awards-overview-2018-02-14.csv&X-Amz-Signature=32fd4fa5968ede2484855fd3537c443a761c5cb9fc911e85ba67ad9e08a3bf76",
						"start_date": "2018-01-01",
						"end_date": "2018-02-01"}
					],
					"pagination": null
				}`))

	client, err := piper.NewClient(twitchclient.ClientConf{
		Host:          "https://piper-service.prod.internal.twitch.tv/",
		BaseTransport: httpmock.DefaultTransport,
	})
	assert.NoError(t, err)

	prop := &models.HelixInputProp{
		UserID:     "testuser",
		Domain:     "games",
		Key:        "\"\"",
		ReportType: "",
		StartDate:  "",
		EndDate:    "",
		Pagination: &models.Pagination{
			Limit:  100,
			Offset: 1,
		},
	}

	resp, err := client.GetAllReportDownloadURLs(context.Background(), prop, &twitchclient.ReqOpts{})

	assert.Equal(t, &models.HelixReportsResp{
		Data: []models.HelixReport{
			{
				ID:        "123",
				Type:      "overview",
				URL:       "https://twitch-piper-reports-dev.s3-us-west-2.amazonaws.com/dynamic/Game1_overview_2018-01-01_2018-02-01.csv?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=ASIAJ4NJJMX7OHHSIL2A%2F20180227%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20180227T202024Z&X-Amz-Expires=60&X-Amz-Security-Token=FQoDYXdzEMX%2F%2F%2F%2F%2F%2F%2F%2F%2F%2FwEaDO6tMLA%2F9SGKudtoJSK3AzUjwQ1Pauvt3wostnfXqmW1KUtPj%2BI0LnaoTbJLRhP4w5oGgYGuz7Ka8ZdnR3bD3zhz%2BtWUXbxSL51lSJG%2F5HUVhI0I%2BvwWP27Z04jS%2FHkNzYtVIJp3TsYPBn1vceE1GRX9yeOpt81qgmkBwwqr6DWWsRN9lTPBTOw0WegMWWQJBozMA%2Blf%2B44CNeGpYAWfoevuiKLyD3W5nUfMwIyKPTJJvPRJwjRhuRO%2BQHkYC7MVWFVTAeXWsv1DmwPs1vfZW1ltMk6rh138ElLaX0S1dscrgm3VCs%2FvJd42hLtZzqjVTaz2Up8BVK58jcQWRs448GZ318H4KYwF1FJ4HqyCl%2FA2DjnjW5qMI8cugluW6EmyEaZQgV8fY0ULNHgyomaGTro96O9A0wAgINHYOZCpx8KhEushNYTvB1ANwWXSzIWrZDVeLGM%2FS3MjC4td3mg1PnCheKzqWH%2FbrciRFynnBku4AKiE5Pe2NGJKt1oz80yY4ULSAAOZtBMNLr3c3y5cAppjVwkdzuIogNIR%2B1bHkZMuwrfTkdyI5tBCu22k%2BKwLuoZdSI4sMEBgwkxKnpOZUILvRXZjOfEo7uvW1AU%3D&X-Amz-SignedHeaders=host&response-content-disposition=attachment%3Bfilename%3DThe%20Game%20Awards-overview-2018-02-14.csv&X-Amz-Signature=32fd4fa5968ede2484855fd3537c443a761c5cb9fc911e85ba67ad9e08a3bf76",
				StartDate: "2018-01-01",
				EndDate:   "2018-02-01",
			},
		},
		Pagination: nil,
	}, resp)
	assert.NoError(t, err)
}

func TestGetModsReports(t *testing.T) {
	httpmock.Activate()
	defer httpmock.DeactivateAndReset()

	httpmock.RegisterResponder("GET", "https://piper-service.prod.internal.twitch.tv/report/mods/testuser",
		httpmock.NewStringResponder(200, `{"reports": [{"id": "111", "report_type": "overview_v1"}, {"id": "222", "report_type": "overview_v1"}]}`))

	client, err := piper.NewClient(twitchclient.ClientConf{
		Host:          "https://piper-service.prod.internal.twitch.tv/",
		BaseTransport: httpmock.DefaultTransport,
	})

	assert.Nil(t, err)
	assert.NotNil(t, client)

	modsReports, err := client.GetReports(context.Background(), "testuser", "mods", &twitchclient.ReqOpts{})

	assert.Nil(t, err)
	assert.NotNil(t, modsReports)
	assert.Len(t, modsReports.Key, 2)

	assert.NotNil(t, modsReports.Key[0])
	assert.Equal(t, "111", modsReports.Key[0].ID)
	assert.Equal(t, "overview_v1", modsReports.Key[0].ReportType)

	assert.NotNil(t, modsReports.Key[1])
	assert.Equal(t, "222", modsReports.Key[1].ID)
	assert.Equal(t, "overview_v1", modsReports.Key[1].ReportType)
}

func TestGetModReportDownloadURL(t *testing.T) {
	httpmock.Activate()
	defer httpmock.DeactivateAndReset()

	httpmock.RegisterResponder("GET", "https://piper-service.prod.internal.twitch.tv/report/mods/testuser/14502/overview_v1/latest",
		httpmock.NewStringResponder(200, `{"url":"https://twitch-piper-reports-dev.s3-us-west-2.amazonaws.com/dynamic/Test_M0d_overview_v1_2018-10-15-2018-11-01_32612242-9e34-4f33-962a-a9717e9c5210.csv?X-Amz-Algorithm=AWS4-HMAC-SHA256\u0026X-Amz-Credential=ASIAIRS44FSDB4CXV6TA%2F20171002%2Fus-west-2%2Fs3%2Faws4_request\u0026X-Amz-Date=20171002T214732Z\u0026X-Amz-Expires=900\u0026X-Amz-Security-Token=FQoDYXdzEPb%2F%2F%2F%2F%2F%2F%2F%2F%2F%2FwEaDOXXZ9NFPXv0MrBghiK3A6qosCwJBOeiglpm%2FGKpoHV6j9uXo0Seo3Xv%2FtfHbMNrMOFZpQgN4xYvPDc9lvQRDbO8Qqz3EiFZT7%2BXdQ1GR1rp0auuhy%2BMd2U7ktmoUPIMV0E%2FBSdRTqI4cdpVBjTTO4EOimuukQReNKwowYkqznMNJoCl1n3whML1yPeSKME12tbnAvfNm%2FLIzWiZHLu3t3ojDWw%2BDJWq%2BkKBdngDFGKTaxwyR%2BL4NyrAqBU0THaooFD%2FXEo0VyNrky1OzYjCsUVs%2B1WCuA1NyZRmXK30lKEahz7ae1dHOGII9Gj9dnchVoNBiWH%2FoqacW9w2XbPqFur4RE0QgM3%2BMFOhU94vPHPDT1MqEy5YV6koH%2B%2BdNdWJcCbVtIl0JIriXjY2ctlr09KVF%2B5rw3jsrq%2BqtNscKhLxLfim6ryi1b9hJl6ta%2FNHtDorhdrzubdoXunMRNwpjtGkun4Fn3Efbruyltk%2FzIFhJcDOXRLvye%2F2vubxqy7LZDCE9qPICLzlprnKHprHWdefLnlTv0p%2BRYbz6gUFiX%2BptHktZMbvBiihuezEVdRJI4V2I2jd5Jt7G4c4T6EXSVLj98XfuSIouMLKzgU%3D\u0026X-Amz-SignedHeaders=host\u0026X-Amz-Signature=530f824764fa365cba7fce11386a897709052f3d1600028e3052cd675dcf4758"}`))

	client, err := piper.NewClient(twitchclient.ClientConf{
		Host:          "https://piper-service.prod.internal.twitch.tv/",
		BaseTransport: httpmock.DefaultTransport,
	})

	assert.Nil(t, err)
	assert.NotNil(t, client)

	url, err := client.GetReportUrlLatest(context.Background(), "testuser", "mods", "14502", "overview_v1", &twitchclient.ReqOpts{})

	assert.Nil(t, err)
	assert.Equal(t,
		"https://twitch-piper-reports-dev.s3-us-west-2.amazonaws.com/dynamic/Test_M0d_overview_v1_2018-10-15-2018-11-01_32612242-9e34-4f33-962a-a9717e9c5210.csv?X-Amz-Algorithm=AWS4-HMAC-SHA256\u0026X-Amz-Credential=ASIAIRS44FSDB4CXV6TA%2F20171002%2Fus-west-2%2Fs3%2Faws4_request\u0026X-Amz-Date=20171002T214732Z\u0026X-Amz-Expires=900\u0026X-Amz-Security-Token=FQoDYXdzEPb%2F%2F%2F%2F%2F%2F%2F%2F%2F%2FwEaDOXXZ9NFPXv0MrBghiK3A6qosCwJBOeiglpm%2FGKpoHV6j9uXo0Seo3Xv%2FtfHbMNrMOFZpQgN4xYvPDc9lvQRDbO8Qqz3EiFZT7%2BXdQ1GR1rp0auuhy%2BMd2U7ktmoUPIMV0E%2FBSdRTqI4cdpVBjTTO4EOimuukQReNKwowYkqznMNJoCl1n3whML1yPeSKME12tbnAvfNm%2FLIzWiZHLu3t3ojDWw%2BDJWq%2BkKBdngDFGKTaxwyR%2BL4NyrAqBU0THaooFD%2FXEo0VyNrky1OzYjCsUVs%2B1WCuA1NyZRmXK30lKEahz7ae1dHOGII9Gj9dnchVoNBiWH%2FoqacW9w2XbPqFur4RE0QgM3%2BMFOhU94vPHPDT1MqEy5YV6koH%2B%2BdNdWJcCbVtIl0JIriXjY2ctlr09KVF%2B5rw3jsrq%2BqtNscKhLxLfim6ryi1b9hJl6ta%2FNHtDorhdrzubdoXunMRNwpjtGkun4Fn3Efbruyltk%2FzIFhJcDOXRLvye%2F2vubxqy7LZDCE9qPICLzlprnKHprHWdefLnlTv0p%2BRYbz6gUFiX%2BptHktZMbvBiihuezEVdRJI4V2I2jd5Jt7G4c4T6EXSVLj98XfuSIouMLKzgU%3D\u0026X-Amz-SignedHeaders=host\u0026X-Amz-Signature=530f824764fa365cba7fce11386a897709052f3d1600028e3052cd675dcf4758",
		url)
}

func TestGetQuarterStartRange(t *testing.T) {
	httpmock.Activate()
	defer httpmock.DeactivateAndReset()

	httpmock.RegisterResponder("GET", "https://piper-service.prod.internal.twitch.tv/onepager/games/testuser/12345/quarterstartrange",
		httpmock.NewStringResponder(200, `{"max_quarter":"2", "max_year":"2018", "min_quarter":"1", "min_year":"2017"}`))

	client, err := piper.NewClient(twitchclient.ClientConf{
		Host:          "https://piper-service.prod.internal.twitch.tv/",
		BaseTransport: httpmock.DefaultTransport,
	})

	assert.Nil(t, err)
	assert.NotNil(t, client)

	quarterStartRange, err := client.GetQuarterStartRange(context.Background(), "testuser", "12345", false, &twitchclient.ReqOpts{})

	spew.Dump(quarterStartRange)

	assert.Nil(t, err)
	assert.NotNil(t, quarterStartRange)
	assert.Equal(t, quarterStartRange.MaxQuarter, "2")
	assert.Equal(t, quarterStartRange.MaxYear, "2018")
	assert.Equal(t, quarterStartRange.MinQuarter, "1")
	assert.Equal(t, quarterStartRange.MinYear, "2017")
}

func TestGetQuarterStartRangeFailure(t *testing.T) {
	httpmock.Activate()
	defer httpmock.DeactivateAndReset()

	httpmock.RegisterResponder("GET", "https://piper-service.prod.internal.twitch.tv/onepager/games/testuser/12345/quarterstartrange",
		httpmock.NewStringResponder(404, `{"status":404,"message":"Game Not Found","error":"Not Found"}`))

	client, err := piper.NewClient(twitchclient.ClientConf{
		Host:          "https://piper-service.prod.internal.twitch.tv/",
		BaseTransport: httpmock.DefaultTransport,
	})

	assert.Nil(t, err)
	assert.NotNil(t, client)

	quarterStartRange, err := client.GetQuarterStartRange(context.Background(), "testuser", "12345", false, &twitchclient.ReqOpts{})

	spew.Dump(quarterStartRange)

	assert.Error(t, err)
	assert.NotNil(t, quarterStartRange)
	assert.Equal(t, quarterStartRange.MaxQuarter, "")
	assert.Equal(t, quarterStartRange.MaxYear, "")
	assert.Equal(t, quarterStartRange.MinQuarter, "")
	assert.Equal(t, quarterStartRange.MinYear, "")
}

func TestDefaultGetQuarterStartRange(t *testing.T) {
	httpmock.Activate()
	defer httpmock.DeactivateAndReset()

	httpmock.RegisterResponder("GET", "https://piper-service.prod.internal.twitch.tv/onepager/games/testuser/12345/quarterstartrange?default=true",
		httpmock.NewStringResponder(200, `{"max_quarter":"2", "max_year":"2018", "min_quarter":"1", "min_year":"2015"}`))

	client, err := piper.NewClient(twitchclient.ClientConf{
		Host:          "https://piper-service.prod.internal.twitch.tv/",
		BaseTransport: httpmock.DefaultTransport,
	})

	assert.Nil(t, err)
	assert.NotNil(t, client)

	quarterStartRange, err := client.GetQuarterStartRange(context.Background(), "testuser", "12345", true, &twitchclient.ReqOpts{})

	spew.Dump(quarterStartRange)

	assert.Nil(t, err)
	assert.NotNil(t, quarterStartRange)
	assert.Equal(t, quarterStartRange.MaxQuarter, "2")
	assert.Equal(t, quarterStartRange.MaxYear, "2018")
	assert.Equal(t, quarterStartRange.MinQuarter, "1")
	assert.Equal(t, quarterStartRange.MinYear, "2015")
}
