package main

import (
	"bufio"
	"fmt"
	"log"
	"os"
	"os/exec"
	"path/filepath"
	"strings"

	"github.com/pkg/errors"
)

func main() {
	if err := run(); err != nil {
		log.Fatal(err)
	}
}

var (
	profilesByEnvironment = map[string]string{
		"staging": "twitch-service-piper-dev",
		"prod":    "twitch-service-piper-aws",
	}
	listToUnset = []string{"ROLLBAR_API_TOKEN"}
)

func run() error {
	if len(os.Args) < 2 {
		return errors.New("expected environment as argument")
	}
	env := os.Args[1]
	var pathToExec string
	if len(os.Args) == 3 {
		pathToExec = os.Args[2]
	}

	profile, ok := profilesByEnvironment[env]
	if !ok {
		return errors.New("unknown environment provided")
	}

	configFile, err := os.Open(filepath.Join("deploy", "config", env+".config"))
	if err != nil {
		return err
	}

	defer func() {
		if err := configFile.Close(); err != nil {
			log.Println("failed to close deploy file")
		}
	}()

	environmentVariables := []string{
		// needed by piper-service
		"AWS_PROFILE=" + profile,
		"ENVIRONMENT=" + env,
		// needed by go
		"GOPATH=" + os.Getenv("GOPATH"),
		"HOME=" + os.Getenv("HOME"),
	}

	if pathToExec != "" {
		environmentVariables = []string{
			"ENVIRONMENT=" + env,
			"AWS_ACCESS_KEY_ID=" + os.Getenv("AWS_ACCESS_KEY_ID"),
			"AWS_SECRET_ACCESS_KEY=" + os.Getenv("AWS_SECRET_ACCESS_KEY"),
			"AWS_SESSION_TOKEN=" + os.Getenv("AWS_SESSION_TOKEN"),
		}
	}

	scanner := bufio.NewScanner(configFile)
LINE_LOOP:
	for scanner.Scan() {
		line := scanner.Text()

		if !strings.HasPrefix(line, "    ") {
			continue
		}

		line = strings.TrimSpace(line)
		nameValue := strings.SplitN(line, ":", 2)
		if len(nameValue) != 2 {
			return errors.Errorf("failed to parse config line: %s", line)
		}

		for _, toUnset := range listToUnset {
			if toUnset == nameValue[0] {
				continue LINE_LOOP
			}
		}

		environmentVariables = append(environmentVariables, fmt.Sprintf("%s=%s", nameValue[0], strings.Trim(nameValue[1], "\" ")))
	}

	cmdToRun := "go run main.go"
	if pathToExec != "" {
		cmdToRun = pathToExec
	}

	fmt.Printf("running \"%s\"\n", cmdToRun)
	cmdSplit := strings.Split(cmdToRun, " ")

	cmd := exec.Command(cmdSplit[0], cmdSplit[1:]...)
	cmd.Env = environmentVariables

	cmd.Stdout = os.Stdout
	cmd.Stderr = os.Stderr

	return cmd.Run()
}
