package discovery

import (
	"strconv"

	"golang.org/x/net/context"

	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/insights/piper-service/internal/clients/discovery/twitchcategorydb"
	"code.justin.tv/insights/piper-service/models"
)

// go:generate mockery -name Client
type Client interface {
	GetGameNameByID(ctx context.Context, id string) (string, error)
	GetGameByID(ctx context.Context, id string) (models.TopGame, error)
}

type clientImpl struct {
	twirpCli twitchcategorydb.TwitchCategoryDB
}

func NewDiscoveryClient(conf twitchclient.ClientConf) Client {
	httpCli := twitchclient.NewHTTPClient(conf)
	twirpCli := twitchcategorydb.NewTwitchCategoryDBProtobufClient(conf.Host, httpCli)
	return &clientImpl{twirpCli: twirpCli}
}

func (c *clientImpl) GetGameNameByID(ctx context.Context, id string) (string, error) {
	intID, err := strconv.ParseInt(id, 10, 32)
	if err != nil {
		return "", models.ErrGameNotFound
	}

	resp, err := c.twirpCli.GetCategoryByID(ctx, &twitchcategorydb.GetCategoryByIDRequest{Id: int32(intID)})
	if err != nil || resp.Category == nil || resp.Category.Name == "" {
		return "", models.ErrGameNotFound
	}

	return resp.Category.Name, nil
}

func (c *clientImpl) GetGameByID(ctx context.Context, id string) (models.TopGame, error) {
	game := models.TopGame{Id: id}
	name, err := c.GetGameNameByID(ctx, id)
	game.Name = name
	return game, err
}
