package elerium

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/url"

	"code.justin.tv/chat/golibs/logx"

	"golang.org/x/net/context"

	"code.justin.tv/foundation/twitchclient"
	piperconfig "code.justin.tv/insights/piper-service/internal/config"
	"code.justin.tv/insights/piper-service/models"
)

const (
	modsKeyPrefix = "mods_user_"
)

func (c *clientImpl) GetModsFromClient(ctx context.Context, userID string, config piperconfig.PiperConfig) ([]models.Mod, error) {
	userID = url.QueryEscape(userID)

	path := (&url.URL{
		Path: "/api/internal/insights/" + userID,
	}).String()

	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return nil, err
	}

	req.Header.Set(config.ModsEleriumApiKey, config.ModsEleriumApiToken)

	resp, err := c.Do(ctx, req, twitchclient.ReqOpts{StatName: "piper_service.get_mods_by_userid"})
	if err != nil {
		return nil, err
	}

	defer func() {
		err = resp.Body.Close()
	}()

	switch resp.StatusCode {
	case http.StatusOK:
		var modsResp models.ModResp

		body, err := ioutil.ReadAll(resp.Body)
		if err != nil {
			return nil, err
		}
		if err := json.Unmarshal(body, &modsResp); err != nil {
			return nil, err
		}
		return modsResp.Mod, nil
	case http.StatusNotFound:
		return nil, models.ErrUserNoMods
	default:
		return nil, models.ErrInternalError
	}
}

func (c *clientImpl) GetModsByUserID(ctx context.Context, userID string, config piperconfig.PiperConfig) (*[]models.Mod, error) {
	var eleriumRespString string

	key := modsKeyPrefix + userID

	eleriumRespString, found := c.Cacher.GetStringProperties(ctx, key)

	if found {
		eleriumRespCached := []models.Mod{}
		err := json.Unmarshal([]byte(eleriumRespString), &eleriumRespCached)
		if err != nil {
			return nil, err
		}
		return &eleriumRespCached, nil
	}

	eleriumResp, err := c.GetModsFromClient(ctx, userID, config)
	if err != nil {
		return nil, err
	}

	if eleriumResp == nil {
		return nil, models.ErrUserNoMods
	}

	eleriumRespJSON, err := json.Marshal(eleriumResp)
	if err != nil {
		return nil, err
	}
	err = c.Cacher.CacheStringProperties(ctx, key, string(eleriumRespJSON))
	if err != nil {
		logx.Error(ctx, fmt.Sprintf("failed to call get extensions for %s in cache", key))
	}

	return &eleriumResp, nil
}
