package owl

import (
	"context"
	"fmt"
	"strings"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/insights/piper-service/backend"
)

const (
	keyDelim                = "|"
	extensionID_key_prefix  = "eID_"
	extensionIDs_key_prefix = "eIDs_"
)

// CachedClient wraps a Client with caching
type CachedClient struct {
	Wrapped Client
	Cacher  backend.Cacher
}

// GetExtensionsByUserID is cached
func (c *CachedClient) GetExtensionsByUserID(ctx context.Context, userID string) ([]string, error) {
	var extensionIDsString string

	key := extensionIDs_key_prefix + userID
	extensionIDsString, found := c.Cacher.GetStringProperties(ctx, key)

	if found {
		extensionIDs := strings.Split(extensionIDsString, keyDelim)
		return extensionIDs, nil
	}

	extensionIDs, err := c.Wrapped.GetExtensionsByUserID(ctx, userID)

	extensionIDsString = strings.Join(extensionIDs, keyDelim)
	err = c.Cacher.CacheStringProperties(ctx, key, extensionIDsString)
	if err != nil {
		logx.Error(ctx, fmt.Sprintf("failed to call get extensions for %s in cache", key))
	}

	return extensionIDs, nil
}

// GetExtensionName is cached
func (c *CachedClient) GetExtensionName(ctx context.Context, extensionID string) string {
	var domainName string

	key := extensionID_key_prefix + extensionID
	domainName, found := c.Cacher.GetStringProperties(ctx, key)
	if found {
		return domainName
	}

	domainName = c.Wrapped.GetExtensionName(ctx, extensionID)

	err := c.Cacher.CacheStringProperties(ctx, key, domainName)
	if err != nil {
		logx.Error(ctx, fmt.Sprintf("failed to call get extension name for %s in cache", key))
	}

	return domainName
}

// IsClientIDUserMatching is not cached.
func (c *CachedClient) IsClientIDUserMatching(ctx context.Context, userID string, clientID string) (bool, error) {
	return c.Wrapped.IsClientIDUserMatching(ctx, userID, clientID)
}

// IsValidExtension is not cached.
func (c *CachedClient) IsValidExtension(ctx context.Context, extensionID string) error {
	return c.Wrapped.IsValidExtension(ctx, extensionID)
}
