package owl

import (
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"github.com/stretchr/testify/suite"
	"golang.org/x/net/context"

	"code.justin.tv/insights/piper-service/backend/mocks"
	owlmock "code.justin.tv/insights/piper-service/internal/clients/owl/mocks"

	"github.com/stretchr/testify/mock"
)

const (
	userID = "12331"
)

type CacheTest struct {
	suite.Suite
	mockClient *owlmock.Client
	mockCacher *mocks.Cacher
	client     Client
	ctx        context.Context
}

func (suite *CacheTest) SetupTest() {
	var err error
	suite.mockClient = &owlmock.Client{}
	suite.mockCacher = &mocks.Cacher{}

	suite.client = &CachedClient{
		Wrapped: suite.mockClient,
		Cacher:  suite.mockCacher,
	}

	suite.ctx = context.Background()
	assert.NoError(suite.T(), err)
}

func (suite *CacheTest) TearDownTest() {
	suite.mockClient.AssertExpectations(suite.T())
	suite.mockCacher.AssertExpectations(suite.T())
}

func (suite *CacheTest) TestUtilsSuite_GetExtensionsByUserIDReturnsFromCache() {
	suite.mockCacher.On("GetStringProperties", mock.Anything, extensionIDs_key_prefix+userID).Return("1212|1213", true)

	extensionIds, err := suite.client.GetExtensionsByUserID(suite.ctx, userID)

	assert.NoError(suite.T(), err)
	require.NotNil(suite.T(), extensionIds)
	require.Len(suite.T(), extensionIds, 2)
}

func (suite *CacheTest) TestUtilsSuite_GetExtensionsByUserIDSavesToCache() {

	suite.mockCacher.On("GetStringProperties", mock.Anything, extensionIDs_key_prefix+userID).Return("", false)
	suite.mockCacher.On("CacheStringProperties", mock.Anything, extensionIDs_key_prefix+userID, "1212|1213").Return(nil)

	suite.mockClient.On("GetExtensionsByUserID", mock.Anything, "12331").Return([]string{"1212", "1213"}, nil)

	extensionIds, err := suite.client.GetExtensionsByUserID(suite.ctx, userID)
	assert.NoError(suite.T(), err)
	require.NotNil(suite.T(), extensionIds)
	require.Len(suite.T(), extensionIds, 2)
}

func (suite *CacheTest) TestUtilsSuite_GetExtensionsByUserIDSavesEmptyListToCache() {

	suite.mockCacher.On("GetStringProperties", mock.Anything, extensionIDs_key_prefix+userID).Return("", false)
	suite.mockCacher.On("CacheStringProperties", mock.Anything, extensionIDs_key_prefix+userID, "").Return(nil)

	suite.mockClient.On("GetExtensionsByUserID", mock.Anything, "12331").Return(nil, nil)

	extensionIds, err := suite.client.GetExtensionsByUserID(suite.ctx, userID)
	assert.NoError(suite.T(), err)
	require.Nil(suite.T(), extensionIds)
	require.Len(suite.T(), extensionIds, 0)
}

func TestUtilsSuite(t *testing.T) {
	suite.Run(t, new(CacheTest))
}
