package owl

import (
	"context"
	"errors"
	"fmt"
	"regexp"
	"strconv"
	"strings"

	"code.justin.tv/chat/golibs/logx"
	owl "code.justin.tv/web/owl/client"
)

var ErrClientOwnedByOrg = errors.New("not a personal client")
var ErrForbiddenClientID = errors.New("forbidden client id")

func (c *clientImpl) IsClientIDUserMatching(ctx context.Context, userID string, clientID string) (bool, error) {
	clientInfo, err := c.owl.GetClient(ctx, clientID, nil)
	if err == owl.ErrForbiddenClientID || err == owl.ErrInvalidClientID {
		return false, ErrForbiddenClientID
	}
	if clientInfo == nil || err != nil || clientInfo.OwnerID == nil {
		return false, err
	}

	// clientInfo.GroupID is set when the client is owned by an organization
	if clientInfo.GroupID != nil && *clientInfo.GroupID != "" {
		return false, ErrClientOwnedByOrg
	}

	intID, err := strconv.Atoi(userID)
	if err != nil {
		return false, err
	}

	return intID == *clientInfo.OwnerID, nil
}

func (c *clientImpl) GetExtensionName(ctx context.Context, clientID string) string {
	var domainName string
	clientInfo, err := c.owl.GetClient(ctx, clientID, nil)
	if err != nil {
		domainName = clientID
		logx.Error(ctx, fmt.Sprintf("Couldn't retrieve user client info: %v", err))
	} else {
		domainName = getExtensionName(clientInfo.Name)
	}
	return domainName
}

func (c *clientImpl) GetExtensionsByUserID(ctx context.Context, userID string) ([]string, error) {
	var extensionIDs []string

	filter := getClientsFilter(userID)
	cursor := ""

	for {
		extensions, nextCursor, err := c.owl.GetClients(ctx, cursor, false, true, filter, "client_id", "ASC", nil)

		if err != nil {
			logx.Error(ctx, fmt.Sprintf("Couldn't retrieve user extensions: %v", err))
			return nil, err
		}

		for _, extension := range extensions {
			extensionIDs = append(extensionIDs, extension.ClientID)
		}

		if nextCursor != "" {
			cursor = nextCursor
		} else {
			break
		}
	}

	return extensionIDs, nil
}

func (c *clientImpl) IsValidExtension(ctx context.Context, extensionID string) error {
	if extensionID == "" {
		return nil
	}
	_, err := c.owl.GetClient(ctx, extensionID, nil)
	return err
}

func getClientsFilter(userID string) map[owl.FilterableColumn]string {
	return map[owl.FilterableColumn]string{
		owl.FilterColOwnerID: userID,
	}
}

func getExtensionName(clientName string) string {
	elems := strings.Split(clientName, " ")
	if len(elems) > 2 {
		if regexp.MustCompile(`^[0-9]`).MatchString(elems[len(elems)-1]) {
			elems = elems[:len(elems)-1]
		}
		if elems[0] == "Extension" {
			elems = elems[1:]
		}
	}
	return strings.Join(elems, " ")
}
