package owl

import (
	"context"
	"testing"

	"github.com/stretchr/testify/require"

	owlMocks "code.justin.tv/insights/piper-service/internal/clients/owl/mocks"
	owl "code.justin.tv/web/owl/client"
	"code.justin.tv/web/owl/oauth2"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type getSuite struct {
	suite.Suite
	mockOwl *owlMocks.OwlClient

	client Client
}

func TestRunGetSuite(t *testing.T) {
	suite.Run(t, new(getSuite))
}

func (suite *getSuite) SetupTest() {
	suite.mockOwl = &owlMocks.OwlClient{}
	suite.client = &clientImpl{
		owl: suite.mockOwl,
	}
}

func (suite *getSuite) TearDownTest() {
	suite.mockOwl.AssertExpectations(suite.T())
}

func (suite *getSuite) TestGetExtensionsByUserID() {
	suite.mockOwl.On("GetClients", mock.Anything, "", false, true, map[owl.FilterableColumn]string{"owner_id": "12331"}, mock.Anything, mock.Anything, mock.Anything).Return([]*oauth2.Client{{ClientID: "1212"}}, "1234567", nil)
	suite.mockOwl.On("GetClients", mock.Anything, "1234567", false, true, map[owl.FilterableColumn]string{"owner_id": "12331"}, mock.Anything, mock.Anything, mock.Anything).Return([]*oauth2.Client{{ClientID: "1213"}}, "", nil)

	extensionIDs, err := suite.client.GetExtensionsByUserID(context.Background(), "12331")
	suite.Require().NoError(err)
	suite.Require().Equal([]string{"1212", "1213"}, extensionIDs)
}

func (suite *getSuite) TestIsClientIDUserMatching_OwnerIDMatch() {
	suite.mockOwl.On("GetClient", mock.Anything, "1", mock.Anything).Return(&oauth2.Client{ClientID: "1212", OwnerID: ptrInt(12331)}, nil)

	hasAccess, err := suite.client.IsClientIDUserMatching(context.Background(), "12331", "1")
	suite.Require().NoError(err)
	suite.Require().True(hasAccess)
}

func (suite *getSuite) TestIsClientIDUserMatching_OwnerIDMatch_GroupOwned() {
	suite.mockOwl.On("GetClient", mock.Anything, "1", mock.Anything).Return(&oauth2.Client{ClientID: "1212", OwnerID: ptrInt(12331), GroupID: ptrString("123")}, nil)

	hasAccess, err := suite.client.IsClientIDUserMatching(context.Background(), "12331", "1")
	suite.Require().Error(err)
	suite.Require().Equal(err, ErrClientOwnedByOrg)
	suite.Require().False(hasAccess)
}

func (suite *getSuite) TestIsClientIDUserMatching_OwnerIDMatch_GroupEmptyString() {
	suite.mockOwl.On("GetClient", mock.Anything, "1", mock.Anything).Return(&oauth2.Client{ClientID: "1212", OwnerID: ptrInt(12331), GroupID: ptrString("")}, nil)

	hasAccess, err := suite.client.IsClientIDUserMatching(context.Background(), "12331", "1")
	suite.Require().NoError(err)
	suite.Require().True(hasAccess)
}

func ptrString(s string) *string {
	return &s
}

func ptrInt(i int) *int {
	return &i
}

func TestExtensionName(t *testing.T) {
	actual := getExtensionName("Extension WoW Armory 1506037411")
	require.Equal(t, "WoW Armory", actual)
}
