package piperdb

import "github.com/lib/pq"

type dbQuery struct {
	QueryName string
	Query     string
	Args      []interface{}
}

func newDBQquery(name, query string, args ...interface{}) dbQuery {
	return dbQuery{
		QueryName: name,
		Query:     query,
		Args:      args,
	}
}

func buildGamesOverviewReportQuery(gameName, startDate, endDate string) dbQuery {
	name := "game_overview_v1_" + gameName + "_" + startDate + "_" + endDate

	query := `
	SELECT cal_date::date as "Date",
		$1 AS "Game",
		COALESCE(live_views, 0) as "Live Views",
		COALESCE(non_live_views, 0) as "Not-Live Views",
		COALESCE(live_uniques, 0) as "Live Unique Viewers",
		COALESCE(non_live_uniques, 0) as "Not-Live Unique Viewers",
		COALESCE(live_hours, 0) as "Live Hours Watched",
		COALESCE(non_live_hours, 0) as "Not-Live Hours Watched",
		COALESCE(unique_broadcasters, 0) as "Unique Broadcasters",
		COALESCE(hours_broadcast, 0) as "Hours Broadcast"
	FROM (
		SELECT DISTINCT DATE_TRUNC('day', date) cal_date
		FROM dim_calendar
		WHERE date >= $2 AND date <= $3
	) date_dim
	LEFT JOIN games_overview g
	ON cal_date = record_date and game = $1
	ORDER BY cal_date`

	return newDBQquery(name, query, gameName, startDate, endDate)
}

func buildExtensionsOverviewReportQuery(extensionID, startDate, endDate string) dbQuery {
	name := "extension_overview_v1_" + extensionID + "_" + startDate + "_" + endDate

	query := `
	SELECT date::date AS "Date"
		, lower(rxl.extension_name) AS "Extension Name"
		, eo.extension_id AS "Extension Client ID"
		, COALESCE(installs, 0) AS "Installs"
		, COALESCE(uninstalls, 0) AS "Uninstalls"
		, COALESCE(activates, 0) AS "Activations"
		, COALESCE(unique_active_channels,0)  AS "Unique Active Channels"
		, COALESCE(total_renders,0)  AS "Renders"
		, COALESCE(unique_renderers,0)  AS "Unique Renderers"
		, COALESCE(total_views, 0) AS "Views"
		, COALESCE(unique_viewers, 0) AS "Unique Viewers"
		, COALESCE(unique_interactors, 0) AS "Unique Interactors"
		, COALESCE(total_clicks,0) AS "Clicks"
		, COALESCE(clicks_per_interactor, 0) AS "Clicks Per Interactor"
		, COALESCE(interaction_rate,0)  AS "Interaction Rate"
	FROM dim_calendar a
	LEFT JOIN extensions_overview eo
	ON date = record_date AND extension_id = $1
	LEFT JOIN released_extensions_latest rxl
	ON eo.extension_id = rxl.extension_id
	WHERE date >= $2 AND date <= $3 AND record_date IS NOT NULL
	ORDER BY 1`

	return newDBQquery(name, query, extensionID, startDate, endDate)
}

func buildGamesOverviewV2ReportQuery(gameName, startDate, endDate string) dbQuery {
	name := "game_overview_v2_" + gameName + "_" + startDate + "_" + endDate

	query := `
	SELECT cal_date::date AS "Date",
		$1 AS "Game",
		game_id AS "Game ID",
		COALESCE(live_views, 0) + COALESCE(non_live_views, 0) as "Total Views",
		COALESCE(live_views, 0) AS "Live Views",
		COALESCE(non_live_views, 0) AS "Not-Live Views",
		COALESCE(total_unique_viewers, 0) AS "Total Unique Viewers",
		COALESCE(live_uniques, 0) AS "Live Unique Viewers",
		COALESCE(non_live_uniques, 0) AS "Not-Live Unique Viewers",
		COALESCE(average_concurrent_viewers, 0) AS "Average Concurrent Viewers",
		COALESCE(peak_concurrent_viewers, 0) AS "Peak Concurrent Viewers",
		CASE WHEN peak_time_concurrent_viewers IS NULL THEN '12:00 AM'
			ELSE TO_CHAR(peak_time_concurrent_viewers::TIME, 'HH12:MI AM')
			END AS "Peak Time - Concurrent Viewers",
		COALESCE(live_hours, 0) + COALESCE(non_live_hours, 0) as "Total Hours Watched",
		COALESCE(live_hours, 0) AS "Live Hours Watched",
		COALESCE(non_live_hours, 0) AS "Not-Live Hours Watched",
		COALESCE(unique_broadcasters, 0) AS "Unique Broadcasters",
		COALESCE(hours_broadcast, 0) AS "Hours Broadcast",
		COALESCE(average_concurrent_broadcasters, 0) AS "Average Concurrent Broadcasters",
		COALESCE(peak_concurrent_broadcasters, 0) AS "Peak Concurrent Broadcasters",
		CASE WHEN peak_time_concurrent_broadcasters IS NULL THEN '12:00 AM'
			ELSE TO_CHAR(peak_time_concurrent_broadcasters::TIME, 'HH12:MI AM')
			END AS "Peak Time - Concurrent Broadcasters",
		COALESCE(live_unique_chat_participants, 0) AS "Live Unique Chat Participants",
		COALESCE(total_live_chat_messages_sent, 0) AS "Total Live Chat Messages Sent",
		COALESCE(unique_active_channels_with_extension, 0) AS "Broadcasters With Active Extensions",
		COALESCE(unique_active_extensions, 0) AS "Unique Active Extensions",
		COALESCE(clips_created, 0) AS "Clips Created",
		COALESCE(clip_views, 0) AS "Clip Views",
		CASE WHEN top_clip_url IS NULL THEN '' 
			WHEN top_clip_url LIKE '%%?%%' THEN CONCAT(top_clip_url, '&tt_medium=dx_insights')
			ELSE CONCAT(top_clip_url, '?tt_medium=dx_insights') END AS "Top Clip URL",
		CASE WHEN top_clip_url_embed IS NULL THEN ''
			WHEN top_clip_url_embed LIKE '%%?%%' THEN CONCAT(top_clip_url_embed, '&tt_medium=dx_insights')
			ELSE CONCAT(top_clip_url_embed, '?tt_medium=dx_insights') END AS "Top Clip URL Embed"
	FROM (
		SELECT DISTINCT DATE_TRUNC('day', date) cal_date
	    FROM dim_calendar
	    WHERE "date" >= $2 AND "date" <= $3
	) date_dim
	LEFT JOIN games_overview_v2 g
	ON cal_date = record_date and game = $1
	WHERE record_date IS NOT NULL
	ORDER BY cal_date`

	return newDBQquery(name, query, gameName, startDate, endDate)
}

func buildExtensionsOverviewV2ReportQuery(extensionID, startDate, endDate string) dbQuery {
	name := "extension_overview_v2_" + extensionID + "_" + startDate + "_" + endDate

	query := `
	SELECT calendar.cal_date::date                                   AS "Date",
		LOWER(e.extension_name)                                 AS "Extension Name",
		$1                                          			AS "Extension Client ID",
		COALESCE(e.ext_page_views, 0)                           AS "Extension Details Page Visits",
		COALESCE(e.ext_page_views_uniq_devices, 0)              AS "Unique Extension Details Page Visits",
		COALESCE(e.installs, 0)                                 AS "Installs",
		COALESCE(e.uninstalls, 0)                               AS "Uninstalls",
		COALESCE(e.activates, 0)                                AS "Activations",
		COALESCE(e.renders_uniq_channels, 0)                    AS "Unique Active Channels",
		COALESCE(e.renders_uniq_channels_7d, 0)                 AS "Unique Active Channels Last 7 Days",
		COALESCE(e.renders_uniq_channels_30d, 0)                AS "Unique Active Channels Last 30 Days",
		COALESCE(e.identity_links_uniq_users, 0)                AS "Unique Identity Links",
		COALESCE(e.identity_unlinks_uniq_users, 0)              AS "Unique Identity Unlinks",
		COALESCE(e.renders, 0)                                  AS "Renders",
		COALESCE(e.renders_uniq_devices, 0)                     AS "Unique Renderers",
		COALESCE(e.renders_uniq_devices_7d, 0)                  AS "Unique Renderers Last 7 Days",
		COALESCE(e.renders_uniq_devices_30d, 0)                 AS "Unique Renderers Last 30 Days",
		COALESCE(e.views, 0)                                    AS "Views",
		COALESCE(e.views_uniq_devices, 0)                       AS "Unique Viewers",
		COALESCE(e.views_uniq_devices_7d, 0)                    AS "Unique Viewers Last 7 Days",
		COALESCE(e.views_uniq_devices_30d, 0)                   AS "Unique Viewers Last 30 Days",
		COALESCE(e.mouseenters, 0)                              AS "Mouseenters",
		COALESCE(e.mouseenters_uniq_devices, 0)                 AS "Unique Mouseenters",
		COALESCE(e.mouseenters_uniq_devices_7d, 0)              AS "Unique Mouseenters Last 7 Days",
		COALESCE(e.mouseenters_uniq_devices_30d, 0)             AS "Unique Mouseenters Last 30 Days",
		ROUND(COALESCE(e.mouseenters::NUMERIC/NULLIF(e.views_uniq_devices,0),0),4) AS "Mouseenters Per Viewer",
		ROUND(COALESCE(e.mouseenters_uniq_devices::NUMERIC/NULLIF(e.views_uniq_devices,0),0),4) AS "Mouseenter Rate",
		COALESCE(e.clicks, 0)                                   AS "Clicks",
		COALESCE(e.clicks_uniq_devices, 0)                      AS "Unique Interactors",
		COALESCE(e.clicks_uniq_devices_7d, 0)                   AS "Unique Interactors Last 7 Days",
		COALESCE(e.clicks_uniq_devices_30d, 0)                  AS "Unique Interactors Last 30 Days",
		ROUND(COALESCE(e.clicks::NUMERIC/NULLIF(e.clicks_uniq_devices,0),0),4) AS "Clicks Per Interactor",
		ROUND(COALESCE(e.clicks_uniq_devices::NUMERIC/NULLIF(e.views_uniq_devices,0),0),4) AS "Interaction Rate",
		COALESCE(e.minimizns, 0)                                AS "Minimizations",
		COALESCE(e.minimizns_uniq_devices, 0)                   AS "Unique Minimizers",
		ROUND(COALESCE(e.minimizns_uniq_devices::NUMERIC/NULLIF(e.views_uniq_devices,0),0),4) AS "Minimization Rate",
		COALESCE(e.unminimizns, 0)                              AS "Unminimizations",
		COALESCE(e.unminimizns_uniq_devices, 0)                 AS "Unique Unminimizers",
		ROUND(COALESCE(e.unminimizns_uniq_devices::NUMERIC/NULLIF(e.views_uniq_devices,0),0),4) AS "Unminimization Rate",
		ROUND(COALESCE(e.bits_spent * 0.002, 0), 2)             AS "Bits Revenue USD",
		COALESCE(e.bits_spent, 0)                               AS "Bits Used",
		COALESCE(e.bits_transactions, 0)                        AS "Bits Transactions",
		ROUND(COALESCE(e.bits_spent::NUMERIC/NULLIF(e.bits_transactions,0),0),4) AS "Bits Per Transaction",
		COALESCE(e.bits_uniq_users, 0)                          AS "Unique Bits Users",
		COALESCE(e.bits_uniq_users_7d, 0)                       AS "Unique Bits Users Last 7 Days",
		COALESCE(e.bits_uniq_users_30d, 0)                      AS "Unique Bits Users Last 30 Days",
		ROUND(COALESCE(e.bits_spent::NUMERIC/NULLIF(e.bits_uniq_users,0),0),4) AS "Bits Used Per User"
	FROM (
		SELECT DISTINCT DATE_TRUNC('day', date) cal_date
		FROM dim_calendar
		WHERE "date" >= $2 AND "date" <= $3
	) calendar
	LEFT JOIN extensions_overview_v2 e ON calendar.cal_date = e."record_date"
		AND e.extension_id = $1
	ORDER BY calendar.cal_date DESC`

	return newDBQquery(name, query, extensionID, startDate, endDate)
}

func buildModsOverviewV1NameQuery(modID string) dbQuery {
	name := "mods_overview_v1_name_" + modID

	query := `SELECT name FROM mods_overview WHERE project_id = $1 ORDER BY record_date DESC LIMIT 1`

	return newDBQquery(name, query, modID)
}

func buildModsOverviewV1ReportQuery(modID string, startDate, endDate string) dbQuery {
	name := "mods_overview_v1_" + modID + "_" + startDate + "_" + endDate

	query := `
	WITH mod_name AS (
		SELECT project_id, name 
		FROM mods_overview
		WHERE project_id = $1 AND name IS NOT NULL
		ORDER BY record_date DESC
		LIMIT 1
	),
	cal_dim AS (
		SELECT DISTINCT DATE_TRUNC('day', date) cal_date
		FROM dim_calendar
		WHERE date >= $2 AND date <= $3
	),
	fill AS (
		SELECT *
		FROM cal_dim
		CROSS JOIN mod_name
	),
	base AS (
		SELECT record_date
			, project_id
			, points
			, historical_download
			, daily_download
			, daily_unique_download
			, daily_twitch_app_download
			, daily_curse_download
		FROM mods_overview
		WHERE project_id = $1
		AND record_date >= $2 AND record_date <= $3
	),
	final AS (
		SELECT cal_date::date AS date
			, fill.project_id
			, fill.name
			, COALESCE(points, 0) AS points
			, CASE WHEN historical_download IS NULL THEN (
				SELECT historical_download
				FROM base
				WHERE record_date < cal_date
				ORDER BY record_date DESC
				LIMIT 1
			) ELSE historical_download END AS historical_download
			, COALESCE(daily_download, 0) AS daily_download
			, COALESCE(daily_unique_download, 0) AS daily_unique_download
			, COALESCE(daily_twitch_app_download, 0) AS daily_twitch_app_download
			, COALESCE(daily_curse_download, 0) AS daily_curse_download
		FROM fill
		LEFT JOIN base b
		ON cal_date = record_date AND b.project_id = $1
	)
	SELECT date as "Date"
		, project_id AS "Project ID"
		, name AS "Name"
		, points AS "Points"
		, historical_download AS "Historical Download"
		, daily_download AS "Daily Download"
		, daily_unique_download AS "Daily Unique Download"
		, daily_twitch_app_download AS "Daily Twitch App Download"
		, daily_curse_download AS "Daily Curse Forge Download"
	FROM final
	WHERE historical_download IS NOT NULL
	ORDER BY 1`

	return newDBQquery(name, query, modID, startDate, endDate)
}

func buildDropsCampaignOverviewReportQuery(campaignID, startDate, endDate string) dbQuery {
	name := "drops_campaigns_overview_" + campaignID + "_" + startDate + "_" + endDate

	query := `
	SELECT calendar.cal_date::date 	as "Date",
		campaign_id					as "Campaign ID",
		COALESCE(campaign_name, '') as "Campaign Name",
		game_id						as "Game ID",
		COALESCE(game_name, '')		as "Game Name",
		streamers_eligible			as "Streamers Eligible",
		viewers_eligible           	as "Viewers Eligible",
		event_drops_earned			as "Event-Based Drops Earned",
		mw_drops_earned				as "Time-Based Drops Earned",
		viewers_claimed            	as "Viewers Claimed",
		viewers_linked_and_claimed	as "Viewers Linked And Claimed",
		streamers_eligible_accum	as "Streamers Eligible (Cumulative)",
		viewers_eligible_accum		as "Viewers Eligible (Cumulative)",
		event_drops_earned_accum	as "Event-Based Drops Earned (Cumulative)",
		mw_drops_earned_accum		as "Time-Based Drops Earned (Cumulative)",
		viewers_claimed_accum		as "Viewers Claimed (Cumulative)",
		viewers_linked_and_claimed_accum as "Viewers Linked and Claimed (Cumulative)"
	FROM (
		SELECT DISTINCT DATE_TRUNC('day', date) cal_date
		FROM dim_calendar
		WHERE "date" >= $2 AND "date" <= $3
	) calendar
	LEFT JOIN drops_campaigns_overview d
		ON calendar.cal_date = d.record_date
	WHERE campaign_id = $1
	ORDER BY 1
	`

	return newDBQquery(name, query, campaignID, startDate, endDate)
}

func buildDropsCampaignDropOverviewReportQuery(campaignID, startDate, endDate string) dbQuery {
	name := "drops_campaigns_drops_overview_" + campaignID + "_" + startDate + "_" + endDate

	query := `
	SELECT calendar.cal_date::date 	as "Date",
		campaign_id					as "Campaign ID",
		COALESCE(campaign_name, '')	as "Campaign Name",
		drop_id						as "Drop ID",
		COALESCE(drop_name, '')		as "Drop Name",
		game_id						as "Game ID",
		COALESCE(game_name, '')		as "Game Name",
		streamers_eligible          as "Streamers Eligible",
		event_drops_earned			as "Event-Based Drops Earned",
		mw_drops_earned				as "Time-Based Drops Earned",
		viewers_claimed            	as "Viewers Claimed",
		viewers_linked_and_claimed	as "Viewers Linked And Claimed"
	FROM (
		SELECT DISTINCT DATE_TRUNC('day', date) cal_date
		FROM dim_calendar
		WHERE "date" >= $2 AND "date" <= $3
	) calendar 
	LEFT JOIN drops_campaigns_drops_overview d
		ON calendar.cal_date = d.record_date
	WHERE campaign_id = $1
	ORDER BY 1
	`

	return newDBQquery(name, query, campaignID, startDate, endDate)
}

func buildDropsCampaignTopStreamersReportQuery(campaignID, startDate, endDate string) dbQuery {
	name := "drops_campaigns_top_streamers_" + campaignID + "_" + startDate + "_" + endDate

	query := `
	SELECT calendar.cal_date::date 	as "Date",
		campaign_id					as "Campaign ID",
		COALESCE(campaign_name, '')	as "Campaign Name",
		COALESCE(channel, '')		as "Channel",
		viewers_eligible           	as "Viewers Eligible",
		event_drops_earned			as "Event-Based Drops Earned",
		mw_drops_earned				as "Time-Based Drops Earned",
		viewers_claimed            	as "Viewers Claimed",
		viewers_linked_and_claimed	as "Viewers Linked And Claimed"
	FROM (
		SELECT DISTINCT DATE_TRUNC('day', date) cal_date
		FROM dim_calendar
		WHERE "date" >= $2 AND "date" <= $3
	) calendar 
	LEFT JOIN drops_campaigns_top_streamers d
		ON calendar.cal_date = d.record_date
	WHERE campaign_id = $1
	ORDER BY 1
	`

	return newDBQquery(name, query, campaignID, startDate, endDate)
}

func buildAvailableDomainReportsQuery(domain, reportType string, domainIDs []string) dbQuery {
	name := "availableDomainReports_" + domain + "_" + reportType

	query := `
	SELECT domain_id, start_date, end_date
	FROM latest_domain_report_meta_data
	WHERE domain = $1 AND report = $2 AND domain_id = ANY($3);`

	return newDBQquery(name, query, domain, reportType, pq.Array(domainIDs))
}
