package piperdb

import (
	"fmt"
	"strings"
	"time"

	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	"code.justin.tv/insights/piper-service/internal/metrics"

	"code.justin.tv/chat/db"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/insights/piper-service/models"
	"golang.org/x/net/context"
)

const (
	shortDate                 = "2006-01-02"
	gamesDomain               = "games"
	extensionsDomain          = "extensions"
	modsDomian                = "mods"
	dropsDomain               = "drops"
	overviewLegacy            = "overview" // only visage endpoint is using overviewLegacy as report type
	overviewV1                = "overview_v1"
	overviewV2                = "overview_v2"
	dropsCampaignOverview     = "campaigns_overview"
	dropsCampaignTopStreamers = "campaigns_top_streamers"
	dropsCampaignDropOverview = "campaigns_drops_overview"
)

func (c *clientImpl) GetAvailableDomainsWithReports(ctx context.Context, domain string, reportType string, domainIDs []string) (result []string, err error) {
	result = make([]string, 0)
	domainReports, err := c.getAvailableDomainReports(ctx, domain, reportType, domainIDs)
	if err != nil || len(domainReports) == 0 {
		return result, err
	}

	for _, x := range domainReports {
		result = append(result, x.DomainID)
	}

	return result, nil
}

func (c *clientImpl) GetDomainReportDateRange(ctx context.Context, domain, reportType string, domainID string) (start_date time.Time, end_date time.Time, err error) {
	start_date = time.Time{}
	end_date = time.Time{}

	domainReports, err := c.getAvailableDomainReports(ctx, domain, reportType, []string{domainID})
	if err != nil || len(domainReports) == 0 {
		return start_date, end_date, models.ErrReportNotFound
	}

	for _, x := range domainReports {
		start_date = x.StartDate
		end_date = x.EndDate
	}

	return start_date, end_date, nil
}

func (c *clientImpl) getAvailableDomainReports(ctx context.Context, domain, reportType string, domainIDs []string) (result []DomainDateRange, err error) {
	reportQuery := buildAvailableDomainReportsQuery(domain, reportType, domainIDs)
	rows, err := c.db.Query(ctx, reportQuery.QueryName, reportQuery.Query, reportQuery.Args...)
	if err != nil {
		return nil, err
	}

	defer func() {
		if cerr := rows.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()

	item := new(DomainDateRange)
	result = make([]DomainDateRange, 0)
	for rows.Next() {
		if err := rows.Scan(&item.DomainID, &item.StartDate, &item.EndDate); err != nil {
			logx.Error(ctx, fmt.Sprintf("Couldn't scan rows from GetAvailableDomainReports: %v", err))
			return nil, models.ErrInternalError
		}
		result = append(result, *item)
	}

	if err := rows.Err(); err != nil {
		return nil, err
	}

	return result, nil
}

func (c *clientImpl) buildReportQuery(ctx context.Context, domain, domain_key, reportType, startDate, endDate string) dbQuery {
	switch {
	case domain == modsDomian && reportType == overviewV1:
		return buildModsOverviewV1ReportQuery(domain_key, startDate, endDate)
	case domain == extensionsDomain && reportType == overviewV2:
		return buildExtensionsOverviewV2ReportQuery(domain_key, startDate, endDate)
	case domain == extensionsDomain && (reportType == overviewV1 || reportType == overviewLegacy):
		return buildExtensionsOverviewReportQuery(domain_key, startDate, endDate)
	case domain == gamesDomain && reportType == overviewV2:
		return buildGamesOverviewV2ReportQuery(strings.ToLower(domain_key), startDate, endDate)
	case domain == dropsDomain && reportType == dropsCampaignOverview:
		return buildDropsCampaignOverviewReportQuery(domain_key, startDate, endDate)
	case domain == dropsDomain && reportType == dropsCampaignTopStreamers:
		return buildDropsCampaignTopStreamersReportQuery(domain_key, startDate, endDate)
	case domain == dropsDomain && reportType == dropsCampaignDropOverview:
		return buildDropsCampaignDropOverviewReportQuery(domain_key, startDate, endDate)
	default:
		return buildGamesOverviewReportQuery(strings.ToLower(domain_key), startDate, endDate)
	}
}

// convertRowsToReports transforms db.Rows to a two dimensional slices with header and scanned data
func convertRowsToReports(ctx context.Context, domain, reportType string, rows db.Rows) ([][]string, error) {
	switch {
	case domain == modsDomian && reportType == overviewV1:
		return scanModsOverviewV1(ctx, rows)
	case domain == extensionsDomain && reportType == overviewV2:
		return scanExtensionsOverviewV2(ctx, rows)
	case domain == extensionsDomain && (reportType == overviewV1 || reportType == overviewLegacy):
		return scanExtensionsOverviewV1(ctx, rows)
	case domain == gamesDomain && reportType == overviewV2:
		return scanGamesOverviewV2(ctx, rows)
	case domain == dropsDomain && reportType == dropsCampaignOverview:
		return scanDropsCampaignOverview(ctx, rows)
	case domain == dropsDomain && reportType == dropsCampaignTopStreamers:
		return scanDropsCampaignTopStreamers(ctx, rows)
	case domain == dropsDomain && reportType == dropsCampaignDropOverview:
		return scanDropsCampaignDropOverview(ctx, rows)
	default:
		return scanGamesOverviewV1(ctx, rows)
	}
}

func collectStatsdMetrics(ctx context.Context, start time.Time, name string) {
	duration := time.Since(start)
	metrics.Reporter().Report(fmt.Sprintf("%s.duration", name), duration.Seconds(), telemetry.UnitSeconds)
}
