package piperdb

import (
	"fmt"
	"strconv"
	"time"

	"code.justin.tv/chat/db"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/insights/piper-service/models"
	"golang.org/x/net/context"
)

func (c *clientImpl) GetDropsReportByDate(ctx context.Context, resourceID, reportType, startDate, endDate string) ([][]string, error) {
	start := time.Now().UTC()

	reportQuery := c.buildReportQuery(ctx, dropsDomain, resourceID, reportType, startDate, endDate)

	rows, err := c.db.Query(ctx, reportQuery.QueryName, reportQuery.Query, reportQuery.Args...)
	if err != nil {
		return nil, err
	}
	defer func() {
		if cerr := rows.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()

	dbResult, err := convertRowsToReports(ctx, dropsDomain, reportType, rows)
	if err != nil {
		return nil, err
	}

	collectStatsdMetrics(ctx, start, "PiperDB.GetDropsReportByDate")

	return dbResult, nil
}

func scanDropsCampaignOverview(ctx context.Context, rows db.Rows) ([][]string, error) {
	result := make([][]string, 0)
	header, err := rows.Columns()
	if err != nil {
		return nil, err
	}
	result = append(result, header)

	for rows.Next() {
		report := new(DropsCampaignsOverviewReport)
		err := rows.Scan(
			&report.Date,
			&report.CampaignID,
			&report.CampaignName,
			&report.GameID,
			&report.GameName,
			&report.StreamersEligible,
			&report.ViewersEligible,
			&report.EventDropsEarned,
			&report.MWDropsEarned,
			&report.ViewersClaimed,
			&report.ViewersLinkedAndClaimed,
			&report.StreamersEligibleCumulative,
			&report.ViewersEligibleCumulative,
			&report.EventDropsEarnedCumulative,
			&report.MWDropsEarnedCumulative,
			&report.ViewersClaimedCumulative,
			&report.ViewersLinkedAndClaimedCumulative,
		)
		if err != nil {
			logx.Error(ctx, fmt.Sprintf("Couldn't scan rows from Drops Campaigns Overview: %v", err))
			return nil, models.ErrInternalError
		}

		data := []string{
			report.Date.Format(shortDate),
			report.CampaignID,
			report.CampaignName,
			report.GameID,
			report.GameName,
			strconv.Itoa(report.StreamersEligible),
			strconv.Itoa(report.ViewersEligible),
			strconv.Itoa(report.EventDropsEarned),
			strconv.Itoa(report.MWDropsEarned),
			strconv.Itoa(report.ViewersClaimed),
			strconv.Itoa(report.ViewersLinkedAndClaimed),
			strconv.Itoa(report.StreamersEligibleCumulative),
			strconv.Itoa(report.ViewersEligibleCumulative),
			strconv.Itoa(report.EventDropsEarnedCumulative),
			strconv.Itoa(report.MWDropsEarnedCumulative),
			strconv.Itoa(report.ViewersClaimedCumulative),
			strconv.Itoa(report.ViewersLinkedAndClaimedCumulative)}

		result = append(result, data)
	}

	if err := rows.Err(); err != nil {
		return nil, err
	}

	return result, nil
}

func scanDropsCampaignDropOverview(ctx context.Context, rows db.Rows) ([][]string, error) {
	result := make([][]string, 0)
	header, err := rows.Columns()
	if err != nil {
		return nil, err
	}
	result = append(result, header)

	for rows.Next() {
		report := new(DropsCampaignsDropsOverviewReport)
		err := rows.Scan(
			&report.Date,
			&report.CampaignID,
			&report.CampaignName,
			&report.DropID,
			&report.DropName,
			&report.GameID,
			&report.GameName,
			&report.StreamersEligible,
			&report.EventDropsEarned,
			&report.MWDropsEarned,
			&report.ViewersClaimed,
			&report.ViewersLinkedAndClaimed,
		)
		if err != nil {
			logx.Error(ctx, fmt.Sprintf("Couldn't scan rows from Drops Campaigns Drops Overview: %v", err))
			return nil, models.ErrInternalError
		}

		data := []string{
			report.Date.Format(shortDate),
			report.CampaignID,
			report.CampaignName,
			report.DropID,
			report.DropName,
			report.GameID,
			report.GameName,
			strconv.Itoa(report.StreamersEligible),
			strconv.Itoa(report.EventDropsEarned),
			strconv.Itoa(report.MWDropsEarned),
			strconv.Itoa(report.ViewersClaimed),
			strconv.Itoa(report.ViewersLinkedAndClaimed)}

		result = append(result, data)
	}

	if err := rows.Err(); err != nil {
		return nil, err
	}

	return result, nil
}

func scanDropsCampaignTopStreamers(ctx context.Context, rows db.Rows) ([][]string, error) {
	result := make([][]string, 0)
	header, err := rows.Columns()
	if err != nil {
		return nil, err
	}
	result = append(result, header)

	for rows.Next() {
		report := new(DropsCampaignsTopStreamersReport)
		err := rows.Scan(
			&report.Date,
			&report.CampaignID,
			&report.CampaignName,
			&report.Channel,
			&report.ViewersEligible,
			&report.EventDropsEarned,
			&report.MWDropsEarned,
			&report.ViewersClaimed,
			&report.ViewersLinkedAndClaimed,
		)
		if err != nil {
			logx.Error(ctx, fmt.Sprintf("Couldn't scan rows from Drops Campaigns Top Streamers: %v", err))
			return nil, models.ErrInternalError
		}

		data := []string{
			report.Date.Format(shortDate),
			report.CampaignID,
			report.CampaignName,
			report.Channel,
			strconv.Itoa(report.ViewersEligible),
			strconv.Itoa(report.EventDropsEarned),
			strconv.Itoa(report.MWDropsEarned),
			strconv.Itoa(report.ViewersClaimed),
			strconv.Itoa(report.ViewersLinkedAndClaimed)}

		result = append(result, data)
	}

	if err := rows.Err(); err != nil {
		return nil, err
	}

	return result, nil
}
