package piperdb

import (
	"fmt"
	"strconv"
	"time"

	"code.justin.tv/chat/db"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/insights/piper-service/models"
	"golang.org/x/net/context"
)

func (c *clientImpl) GetExtensionReportByDate(ctx context.Context, extensionID, reportType, startDate, endDate string) ([][]string, error) {
	start := time.Now().UTC()

	reportQuery := c.buildReportQuery(ctx, extensionsDomain, extensionID, reportType, startDate, endDate)
	rows, err := c.db.Query(ctx, reportQuery.QueryName, reportQuery.Query, reportQuery.Args...)
	if err != nil {
		return nil, err
	}

	defer func() {
		if cerr := rows.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()

	dbresult, err := convertRowsToReports(ctx, extensionsDomain, reportType, rows)

	if err != nil {
		return dbresult, err
	}

	collectStatsdMetrics(ctx, start, "PiperDB.GetExtensionReportByDate")

	return dbresult, nil
}

func scanExtensionsOverviewV1(ctx context.Context, rows db.Rows) (result [][]string, err error) {
	result = make([][]string, 0)
	header, err := rows.Columns()
	if err != nil {
		return nil, err
	}
	result = append(result, header)
	for rows.Next() {
		report := new(ExtensionsOverviewV1Report)
		err := rows.Scan(&report.Date, &report.Extension_name, &report.Extension_client_id, &report.Installs,
			&report.Uninstalls, &report.Activations, &report.Unique_active_channels, &report.Renders, &report.Unique_renderers,
			&report.Views, &report.Unique_viewers, &report.Unique_interactors, &report.Clicks, &report.Clicks_per_interactor, &report.Interaction_rate)
		if err != nil {
			logx.Error(ctx, fmt.Sprintf("Couldn't scan rows from Extensions Overview V1: %v", err))
			return nil, models.ErrInternalError
		}

		data := []string{report.Date.Format(shortDate), report.Extension_name.String, report.Extension_client_id, strconv.Itoa(report.Installs),
			strconv.Itoa(report.Uninstalls), strconv.Itoa(report.Activations), strconv.Itoa(report.Unique_active_channels), strconv.Itoa(report.Renders),
			strconv.Itoa(report.Unique_renderers), strconv.Itoa(report.Views), strconv.Itoa(report.Unique_viewers), strconv.Itoa(report.Unique_interactors),
			strconv.Itoa(report.Clicks), strconv.FormatFloat(report.Clicks_per_interactor, 'f', -1, 64),
			strconv.FormatFloat(report.Interaction_rate, 'f', -1, 64)}

		result = append(result, data)
	}

	if err := rows.Err(); err != nil {
		return nil, err
	}

	return result, nil
}

func scanExtensionsOverviewV2(ctx context.Context, rows db.Rows) (result [][]string, err error) {
	result = make([][]string, 0)
	header, err := rows.Columns()
	if err != nil {
		return nil, err
	}
	result = append(result, header)

	for rows.Next() {
		report := new(ExtensionsOverviewV2Report)
		err := rows.Scan(
			&report.Date,
			&report.Extension_name,
			&report.Extension_client_id,
			&report.Extension_details_page_visits,
			&report.Unique_extension_details_page_visits,
			&report.Installs,
			&report.Uninstalls,
			&report.Activations,
			&report.Unique_active_channels,
			&report.Unique_active_channels_last_7_days,
			&report.Unique_active_channels_last_30_days,
			&report.Unique_identity_links,
			&report.Unique_identity_unlinks,
			&report.Renders,
			&report.Unique_renderers,
			&report.Unique_renderers_last_7_days,
			&report.Unique_renderers_last_30_days,
			&report.Views,
			&report.Unique_viewers,
			&report.Unique_viewers_last_7_days,
			&report.Unique_viewers_last_30_days,
			&report.Mouseenters,
			&report.Unique_mouseenters,
			&report.Unique_mouseenters_last_7_days,
			&report.Unique_mouseenters_last_30_days,
			&report.Mouseenters_per_viewer,
			&report.Mouseenter_rate,
			&report.Clicks,
			&report.Unique_interactors,
			&report.Unique_interactors_last_7_days,
			&report.Unique_interactors_last_30_days,
			&report.Clicks_per_interactor,
			&report.Interaction_rate,
			&report.Minimizations,
			&report.Unique_minimizers,
			&report.Minimization_rate,
			&report.Unminimizations,
			&report.Unique_unminimizers,
			&report.Unminimization_rate,
			&report.Bits_revenue_usd,
			&report.Bits_used,
			&report.Bits_transactions,
			&report.Bits_per_transaction,
			&report.Unique_bits_users,
			&report.Unique_bits_users_last_7_days,
			&report.Unique_bits_users_last_30_days,
			&report.Bits_used_per_user)
		if err != nil {
			logx.Error(ctx, fmt.Sprintf("Couldn't scan rows from Extensions Overview V2: %v", err))
			return nil, models.ErrInternalError
		}

		data := []string{
			report.Date.Format(shortDate),
			report.Extension_name.String,
			report.Extension_client_id,
			strconv.Itoa(report.Extension_details_page_visits),
			strconv.Itoa(report.Unique_extension_details_page_visits),
			strconv.Itoa(report.Installs),
			strconv.Itoa(report.Uninstalls),
			strconv.Itoa(report.Activations),
			strconv.Itoa(report.Unique_active_channels),
			strconv.Itoa(report.Unique_active_channels_last_7_days),
			strconv.Itoa(report.Unique_active_channels_last_30_days),
			strconv.Itoa(report.Unique_identity_links),
			strconv.Itoa(report.Unique_identity_unlinks),
			strconv.Itoa(report.Renders),
			strconv.Itoa(report.Unique_renderers),
			strconv.Itoa(report.Unique_renderers_last_7_days),
			strconv.Itoa(report.Unique_renderers_last_30_days),
			strconv.Itoa(report.Views),
			strconv.Itoa(report.Unique_viewers),
			strconv.Itoa(report.Unique_viewers_last_7_days),
			strconv.Itoa(report.Unique_viewers_last_30_days),
			strconv.Itoa(report.Mouseenters),
			strconv.Itoa(report.Unique_mouseenters),
			strconv.Itoa(report.Unique_mouseenters_last_7_days),
			strconv.Itoa(report.Unique_mouseenters_last_30_days),
			strconv.FormatFloat(report.Mouseenters_per_viewer, 'f', -1, 64),
			strconv.FormatFloat(report.Mouseenter_rate, 'f', -1, 64),
			strconv.Itoa(report.Clicks),
			strconv.Itoa(report.Unique_interactors),
			strconv.Itoa(report.Unique_interactors_last_7_days),
			strconv.Itoa(report.Unique_interactors_last_30_days),
			strconv.FormatFloat(report.Clicks_per_interactor, 'f', -1, 64),
			strconv.FormatFloat(report.Interaction_rate, 'f', -1, 64),
			strconv.Itoa(report.Minimizations),
			strconv.Itoa(report.Unique_minimizers),
			strconv.FormatFloat(report.Minimization_rate, 'f', -1, 64),
			strconv.Itoa(report.Unminimizations),
			strconv.Itoa(report.Unique_unminimizers),
			strconv.FormatFloat(report.Unminimization_rate, 'f', -1, 64),
			strconv.FormatFloat(report.Bits_revenue_usd, 'f', -1, 64),
			strconv.Itoa(report.Bits_used),
			strconv.Itoa(report.Bits_transactions),
			strconv.FormatFloat(report.Bits_per_transaction, 'f', -1, 64),
			strconv.Itoa(report.Unique_bits_users),
			strconv.Itoa(report.Unique_bits_users_last_7_days),
			strconv.Itoa(report.Unique_bits_users_last_30_days),
			strconv.FormatFloat(report.Bits_used_per_user, 'f', -1, 64),
		}

		result = append(result, data)
	}

	if err := rows.Err(); err != nil {
		return nil, err
	}
	return result, nil
}
