package piperdb

import (
	"fmt"
	"strconv"
	"time"

	"code.justin.tv/chat/db"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/insights/piper-service/models"
	"golang.org/x/net/context"
)

func (c *clientImpl) GetGameReportByDate(ctx context.Context, gameName string, reportType, startDate string, endDate string) (r [][]string, err error) {
	start := time.Now().UTC()

	reportQuery := c.buildReportQuery(ctx, gamesDomain, gameName, reportType, startDate, endDate)
	rows, err := c.db.Query(ctx, reportQuery.QueryName, reportQuery.Query, reportQuery.Args...)
	if err != nil {
		return nil, err
	}

	defer func() {
		if cerr := rows.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()

	dbresult, err := convertRowsToReports(ctx, gamesDomain, reportType, rows)
	if err != nil {
		return dbresult, err
	}

	collectStatsdMetrics(ctx, start, "PiperDB.GetGameReportByDate")

	return dbresult, nil
}

func scanGamesOverviewV1(ctx context.Context, rows db.Rows) (result [][]string, err error) {
	result = make([][]string, 0)
	header, err := rows.Columns()
	if err != nil {
		return nil, err
	}
	result = append(result, header)

	for rows.Next() {
		report := new(GamesOverviewV1Report)
		err := rows.Scan(&report.Date, &report.Game, &report.Live_views, &report.Non_live_views, &report.Live_uniques,
			&report.Non_live_uniques, &report.Live_hours, &report.Non_live_hours, &report.Unique_broadcasters, &report.Hours_broadcast)
		if err != nil {
			logx.Error(ctx, fmt.Sprintf("Couldn't scan rows from Games Overview V1: %v", err))
			return nil, models.ErrInternalError
		}

		data := []string{report.Game, report.Date.Format(shortDate), strconv.Itoa(report.Live_views),
			strconv.Itoa(report.Non_live_views), strconv.Itoa(report.Live_uniques), strconv.Itoa(report.Non_live_uniques),
			strconv.FormatFloat(report.Live_hours, 'f', -1, 64),
			strconv.FormatFloat(report.Non_live_hours, 'f', -1, 64),
			strconv.Itoa(report.Unique_broadcasters), strconv.FormatFloat(report.Hours_broadcast, 'f', -1, 64)}

		result = append(result, data)
	}

	if err := rows.Err(); err != nil {
		return nil, err
	}
	return result, nil
}

func scanGamesOverviewV2(ctx context.Context, rows db.Rows) (result [][]string, err error) {
	result = make([][]string, 0)
	header, err := rows.Columns()
	if err != nil {
		return nil, err
	}
	result = append(result, header)

	for rows.Next() {
		report := new(GamesOverviewV2Report)
		err := rows.Scan(&report.Date,
			&report.Game,
			&report.GameID,
			&report.Total_views,
			&report.Live_views,
			&report.Non_live_views,
			&report.Total_unique_viewers,
			&report.Live_uniques,
			&report.Non_live_uniques,
			&report.Average_concurrent_viewers,
			&report.Peak_concurrent_viewers,
			&report.Peak_time_concurrent_viewers,
			&report.Total_hours_watched,
			&report.Live_hours_watched,
			&report.Non_live_hours_watched,
			&report.Unique_broadcasters,
			&report.Hours_broadcast,
			&report.Average_concurrent_broadcasters,
			&report.Peak_concurrent_broadcasters,
			&report.Peak_time_concurrent_broadcasters,
			&report.Live_unique_chat_participants,
			&report.Total_live_chat_messages_sent,
			&report.Unique_active_channels_with_extension,
			&report.Unique_active_extensions,
			&report.Clips_created,
			&report.Clips_views,
			&report.Top_clip_url,
			&report.Top_clip_url_embed,
		)
		if err != nil {
			logx.Error(ctx, fmt.Sprintf("Couldn't scan rows from Games Overview V2: %v", err))
			return nil, models.ErrInternalError
		}

		data := []string{report.Date.Format(shortDate),
			report.Game,
			strconv.Itoa(report.GameID),
			strconv.Itoa(report.Total_views),
			strconv.Itoa(report.Live_views),
			strconv.Itoa(report.Non_live_views),
			strconv.Itoa(report.Total_unique_viewers),
			strconv.Itoa(report.Live_uniques),
			strconv.Itoa(report.Non_live_uniques),
			strconv.Itoa(report.Average_concurrent_viewers),
			strconv.Itoa(report.Peak_concurrent_viewers),
			report.Peak_time_concurrent_viewers,
			strconv.FormatFloat(report.Total_hours_watched, 'f', -1, 64),
			strconv.FormatFloat(report.Live_hours_watched, 'f', -1, 64),
			strconv.FormatFloat(report.Non_live_hours_watched, 'f', -1, 64),
			strconv.Itoa(report.Unique_broadcasters),
			strconv.FormatFloat(report.Hours_broadcast, 'f', -1, 64),
			strconv.Itoa(report.Average_concurrent_broadcasters),
			strconv.Itoa(report.Peak_concurrent_broadcasters),
			report.Peak_time_concurrent_broadcasters,
			strconv.Itoa(report.Live_unique_chat_participants),
			strconv.Itoa(report.Total_live_chat_messages_sent),
			strconv.Itoa(report.Unique_active_channels_with_extension),
			strconv.Itoa(report.Unique_active_extensions),
			strconv.Itoa(report.Clips_created),
			strconv.Itoa(report.Clips_views),
			report.Top_clip_url.String,
			report.Top_clip_url_embed.String,
		}

		result = append(result, data)
	}

	if err := rows.Err(); err != nil {
		return nil, err
	}
	return result, nil
}
