package piperdb

import (
	"fmt"
	"strconv"
	"time"

	"code.justin.tv/chat/db"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/insights/piper-service/models"
	"golang.org/x/net/context"
)

func (c *clientImpl) GetModReportByDate(ctx context.Context, modID, reportType, startDate, endDate string) ([][]string, error) {
	start := time.Now().UTC()

	reportQuery := c.buildReportQuery(ctx, modsDomian, modID, reportType, startDate, endDate)

	rows, err := c.db.Query(ctx, reportQuery.QueryName, reportQuery.Query, reportQuery.Args...)
	if err != nil {
		return nil, err
	}

	defer func() {
		if cerr := rows.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()

	dbresult, err := convertRowsToReports(ctx, modsDomian, reportType, rows)
	if err != nil {
		return dbresult, err
	}

	collectStatsdMetrics(ctx, start, "PiperDB.GetModReportByDate")

	return dbresult, nil
}

func scanModsOverviewV1(ctx context.Context, rows db.Rows) (result [][]string, err error) {
	result = make([][]string, 0)
	header, err := rows.Columns()
	if err != nil {
		return nil, err
	}
	result = append(result, header)

	for rows.Next() {
		report := new(ModsOverviewV1Report)
		err := rows.Scan(&report.Date,
			&report.ModID,
			&report.Name,
			&report.Points,
			&report.Historical_Download,
			&report.Daily_Download,
			&report.Daily_Unique_Download,
			&report.Daily_Twitch_App_Download,
			&report.Daily_Curse_Forge_Download)
		if err != nil {
			logx.Error(ctx, fmt.Sprintf("Couldn't scan rows for Mods Overview: %v", err))
			return nil, models.ErrInternalError
		}

		data := []string{
			report.Date.Format(shortDate),
			strconv.Itoa(report.ModID),
			report.Name,
			strconv.FormatFloat(report.Points, 'f', -1, 64),
			strconv.Itoa(report.Historical_Download),
			strconv.Itoa(report.Daily_Download),
			strconv.Itoa(report.Daily_Unique_Download),
			strconv.Itoa(report.Daily_Twitch_App_Download),
			strconv.Itoa(report.Daily_Curse_Forge_Download)}

		result = append(result, data)
	}

	if err := rows.Err(); err != nil {
		return nil, err
	}
	return result, nil
}

func (c *clientImpl) GetModName(ctx context.Context, modID string) (string, error) {
	start := time.Now().UTC()

	nameQuery := buildModsOverviewV1NameQuery(modID)

	rows, err := c.db.Query(ctx, nameQuery.QueryName, nameQuery.Query, nameQuery.Args...)
	if err != nil {
		return "", err
	}

	defer func() {
		if cerr := rows.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()

	var modName string

	for rows.Next() {
		err := rows.Scan(&modName)
		if err != nil {
			logx.Error(ctx, fmt.Sprintf("Couldn't scan name for Mods ID %s: %v", modID, err))
			return "", models.ErrReportNotFound
		}
	}

	if err := rows.Err(); err != nil {
		return "", err
	}

	collectStatsdMetrics(ctx, start, "PiperDB.GetModName")
	return modName, nil
}
