package piperdb

import (
	"database/sql"
	"strconv"
	"time"

	"code.justin.tv/insights/piper-service/models"
	"golang.org/x/net/context"
)

const (
	sqlSelectDefaultInsights = ` SELECT ` +
		` total_active_extensions      ,` +
		` minutes_watched              ,` +
		` unique_monthly_broadcasters  ,` +
		` unique_daily_visitors        ,` +
		` partnered_streamers          ,` +
		` affiliate_streamers          ,` +
		` total_clips                  ,` +
		` top_three_broadcaster        ,` +
		` top_three_extensions         ,` +
		` top_three_games_by_streamers  ,` +
		` top_three_games_by_viewers    ` +
		` FROM one_pager_topline_default` +
		` WHERE quarter_start = $1`

	sqlSelectInsightsByDateAndGameID = ` SELECT  ` +
		` hours_broadcast_current             	,` +
		` hours_broadcast_qoq                 	,` +
		` hours_watched_current                 ,` +
		` hours_watched_qoq                     ,` +
		` unique_broadcasters_current           ,` +
		` unique_broadcasters_qoq               ,` +
		` unique_viewers_current                ,` +
		` unique_viewers_qoq                    ,` +
		` broadcasting_sessions_current         ,` +
		` broadcasting_sessions_qoq             ,` +
		` viewing_sessions_current              ,` +
		` viewing_sessions_qoq                  ,` +
		` top_channel_login                     ,` +
		` top_channel_displayname               ,` +
		` top_channel_hours_broadcast           ,` +
		` top_channel_hours_watched             ,` +
		` top_channel_avg_ccu                   ,` +
		` rising_channel_login                  ,` +
		` rising_channel_displayname            ,` +
		` rising_channel_hours_broadcast        ,` +
		` rising_channel_hours_watched          ,` +
		` rising_channel_avg_ccu                ,` +
		` most_dedicated_channel_login          ,` +
		` most_dedicated_channel_displayname    ,` +
		` most_dedicated_channel_sessions       ,` +
		` most_dedicated_channel_game_percentage,` +
		` total_chat_messages                   ,` +
		` unique_chatters                       ,` +
		` chatter_percent                       ,` +
		` unique_partner_affiliate              ,` +
		` total_active_extensions               ,` +
		` most_popular_extensions               ,` +
		` drops_received_current                ,` +
		` drops_received_qoq                    ,` +
		` unique_viewers_received_a_drop_current,` +
		` unique_viewers_received_a_drop_qoq    ,` +
		` active_campaigns_current              ,` +
		` active_campaigns_qoq                  ,` +
		` rising_countries_by_streamers         ,` +
		` rising_countries_by_viewers           ,` +
		` top_three_games_by_streamers          ,` +
		` top_three_games_by_viewers             ` +
		` FROM one_pager_topline` +
		` WHERE game_id = $1 AND quarter_start = $2`

	sqlSelectQuarterStartByYearQuarter = `
		SELECT a.quarter_start 
		FROM (
			SELECT MIN(date) quarter_start 
			FROM dim_calendar WHERE year = $1 AND quarter = $2
		) a 
		JOIN one_pager_topline_default b ON a.quarter_start = b.quarter_start;
	`

	sqlSelectQuarterStartRange = `SELECT
		COALESCE(EXTRACT(QUARTER FROM a.max_quarter_start), 0) as max_quarter,
		COALESCE(EXTRACT(QUARTER FROM a.min_quarter_start), 0) as min_quarter,
		COALESCE(EXTRACT(YEAR FROM a.max_quarter_start), 0) as max_year,
		COALESCE(EXTRACT(YEAR FROM a.min_quarter_start), 0) as min_year
		FROM (
			SELECT MAX(quarter_start) as max_quarter_start, MIN(quarter_start) as min_quarter_start  
			FROM one_pager_topline
			WHERE game_id = $1
		) a
	`
	sqlSelectDefaultQuarterStartRange = `SELECT
		COALESCE(EXTRACT(QUARTER FROM a.max_quarter_start), 0) as max_quarter,
		COALESCE(EXTRACT(QUARTER FROM a.min_quarter_start), 0) as min_quarter,
		COALESCE(EXTRACT(YEAR FROM a.max_quarter_start), 0) as max_year,
		COALESCE(EXTRACT(YEAR FROM a.min_quarter_start), 0) as min_year
		FROM (
			SELECT MAX(quarter_start) as max_quarter_start, MIN(quarter_start) as min_quarter_start  
			FROM one_pager_topline_default
		) a
	`
)

func (c *clientImpl) GetGameDefaultOnePagerByQuarterAndYear(ctx context.Context, startTime time.Time) (models.OnePagerDefaultData, error) {
	row := c.db.QueryRow(ctx, "get_default_game_insights_by_date", sqlSelectDefaultInsights, startTime)
	onePagerDefaultData := models.OnePagerDefaultData{}
	if err := row.Scan(
		&onePagerDefaultData.TotalActiveExtensions,
		&onePagerDefaultData.MinutesWatched,
		&onePagerDefaultData.UniqueMonthlyBroadcasters,
		&onePagerDefaultData.UniqueDailyVisitors,
		&onePagerDefaultData.PartneredStreamers,
		&onePagerDefaultData.AffiliateStreamers,
		&onePagerDefaultData.TotalClips,
		&onePagerDefaultData.TopThreeBroadcaster,
		&onePagerDefaultData.TopThreeExtensions,
		&onePagerDefaultData.TopThreeGamesByStreamers,
		&onePagerDefaultData.TopThreeGamesByViewers,
	); err != nil {
		return onePagerDefaultData, err
	}
	return onePagerDefaultData, nil
}

func (c *clientImpl) GetGameOnePagerByQuarterAndYear(ctx context.Context, gameID string, startTime time.Time) (models.OnePagerData, error) {
	onePagerData := models.OnePagerData{}

	gameIDInt, err := strconv.ParseInt(gameID, 10, 32)
	if err != nil {
		return onePagerData, models.ErrInvalidGameID
	}

	row := c.db.QueryRow(ctx, "get_insights_by_date_and_game_id", sqlSelectInsightsByDateAndGameID, gameIDInt, startTime)
	if err := row.Scan(
		&onePagerData.HoursBroadcastCurrent,
		&onePagerData.HoursBroadcastQoQ,
		&onePagerData.HoursWatchedCurrent,
		&onePagerData.HoursWatchedQoQ,
		&onePagerData.UniqueBroadcastersCurrent,
		&onePagerData.UniqueBroadcastersQoQ,
		&onePagerData.UniqueViewersCurrent,
		&onePagerData.UniqueViewersQoQ,
		&onePagerData.BroadcastingSessionsCurrent,
		&onePagerData.BroadcastingSessionsQoQ,
		&onePagerData.ViewingSessionsCurrent,
		&onePagerData.ViewingSessionsQoQ,

		&onePagerData.TopChannelLogin,
		&onePagerData.TopChannelDisplayName,
		&onePagerData.TopChannelHoursBroadcast,
		&onePagerData.TopChannelHoursWatched,
		&onePagerData.TopChannelAvgCCU,
		&onePagerData.RisingChannelLogin,
		&onePagerData.RisingChannelDisplayname,
		&onePagerData.RisingChannelHoursBroadcast,
		&onePagerData.RisingChannelHoursWatched,
		&onePagerData.RisingChannelAvgCCU,
		&onePagerData.MostDedicatedChannelLogin,
		&onePagerData.MostDedicatedChannelDisplayname,
		&onePagerData.MostDedicatedChannelSessions,
		&onePagerData.MostDedicatedChannelGamePercentage,

		&onePagerData.TotalChatMessages,
		&onePagerData.UniqueChatters,
		&onePagerData.ChatterPercent,
		&onePagerData.UniquePartnerAffiliate,

		&onePagerData.TotalActiveExtensions,
		&onePagerData.MostPopularExtensions,
		&onePagerData.DropsReceivedCurrent,
		&onePagerData.DropsReceivedQoQ,
		&onePagerData.UniqueViewersReceivedADropCurrent,
		&onePagerData.UniqueViewersReceivedADropQoQ,
		&onePagerData.ActiveCampaignsCurrent,
		&onePagerData.ActiveCampaignsQoQ,

		&onePagerData.RisingCountriesByStreamers,
		&onePagerData.RisingCountriesByViewers,
		&onePagerData.TopThreeGamesByStreamers,
		&onePagerData.TopThreeGamesByViewers,
	); err != nil {
		if err != sql.ErrNoRows {
			return onePagerData, err
		}
	}
	return onePagerData, nil
}

func (c *clientImpl) GetQuarterStartByYearQuarter(ctx context.Context, year, quarter string) (time.Time, error) {
	var quarterStart time.Time
	row := c.db.QueryRow(ctx, "get_quarter_start_by_year_quarter", sqlSelectQuarterStartByYearQuarter, year, quarter)
	if err := row.Scan(&quarterStart); err != nil {
		if err == sql.ErrNoRows {
			return time.Time{}, models.ErrReportNotFound
		} else {
			return time.Time{}, err
		}
	}
	return quarterStart, nil
}

func (c *clientImpl) GetQuarterStartRange(ctx context.Context, gameID string) (models.QuarterStartRange, error) {
	quarterStartRange := models.QuarterStartRange{}
	gameIDInt, err := strconv.ParseInt(gameID, 10, 32)
	if err != nil {
		return quarterStartRange, models.ErrInvalidGameID
	}

	row := c.db.QueryRow(ctx, "get_quarter_start_range", sqlSelectQuarterStartRange, gameIDInt)

	if err := row.Scan(
		&quarterStartRange.MaxQuarter,
		&quarterStartRange.MinQuarter,
		&quarterStartRange.MaxYear,
		&quarterStartRange.MinYear,
	); err != nil || quarterStartRange.MaxQuarter == "0" {
		return quarterStartRange, models.ErrReportNotFound
	}

	return quarterStartRange, nil
}

func (c *clientImpl) GetDefaultQuarterStartRange(ctx context.Context) (models.QuarterStartRange, error) {
	quarterStartRange := models.QuarterStartRange{}

	row := c.db.QueryRow(ctx, "get_default_quarter_start_range", sqlSelectDefaultQuarterStartRange)

	if err := row.Scan(
		&quarterStartRange.MaxQuarter,
		&quarterStartRange.MinQuarter,
		&quarterStartRange.MaxYear,
		&quarterStartRange.MinYear,
	); err != nil || quarterStartRange.MaxQuarter == "0" {
		return quarterStartRange, models.ErrReportNotFound
	}

	return quarterStartRange, nil
}
