package rbac

import (
	"fmt"
	"strings"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"code.justin.tv/insights/piper-service/backend"
	"golang.org/x/net/context"
)

const (
	userID_games_key_prefix = "user_games_" // user "user_games_{userID}" to store <userID, gameIDs>
	valueDelim              = "|"
)

// CachedClient wraps a Client with caching
type CachedClient struct {
	Wrapped Client
	Cacher  backend.Cacher
}

// GetGameIDsByUserID is cached
func (c *CachedClient) GetGameIDsByUserID(ctx context.Context, userID string) ([]string, error) {
	key := userID_games_key_prefix + userID

	prop, found := c.Cacher.GetStringProperties(ctx, key)
	if found {
		prop = strings.Trim(prop, "\"")
		return strings.Split(prop, valueDelim), nil
	}

	gameIDs, err := c.Wrapped.GetGameIDsByUserID(ctx, userID)
	if err != nil {
		return nil, err
	}

	// could be hystrix timeout, do not cache
	if len(gameIDs) == 0 {
		return gameIDs, err
	}

	err = c.Cacher.CacheStringProperties(ctx, key, strings.Join(gameIDs, valueDelim))
	if err != nil {
		logx.Error(ctx, fmt.Sprintf("failed to call set up games for %s in cache", key))
	}

	return gameIDs, nil
}

// GetCompanyByUserID is not cached
func (c *CachedClient) GetCompanyByUserID(ctx context.Context, userID string) (rbacrpc.Company, error) {
	return c.Wrapped.GetCompanyByUserID(ctx, userID)
}

// Validate is not cached
func (c *CachedClient) Validate(ctx context.Context, params *rbacrpc.ValidateQuery) error {
	return c.Wrapped.Validate(ctx, params)
}
