package rbac

import (
	"context"
	"testing"

	"code.justin.tv/insights/piper-service/backend/mocks"
	rbacClient "code.justin.tv/insights/piper-service/internal/clients/rbac/mocks"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"
	"github.com/stretchr/testify/suite"
)

type CacheTest struct {
	suite.Suite
	mockClient *rbacClient.Client
	mockCacher *mocks.Cacher
	client     Client
	ctx        context.Context
}

const (
	userID    = "1234"
	userGames = "game01|game02"
)

func (suite *CacheTest) SetupTest() {
	var err error

	suite.mockClient = &rbacClient.Client{}
	suite.mockCacher = &mocks.Cacher{}

	suite.client = &CachedClient{
		Wrapped: suite.mockClient,
		Cacher:  suite.mockCacher,
	}

	suite.ctx = context.Background()
	assert.NoError(suite.T(), err)
}

func (suite *CacheTest) TearDownTest() {
	suite.mockClient.AssertExpectations(suite.T())
	suite.mockCacher.AssertExpectations(suite.T())
}

func (suite *CacheTest) TestGetGameIDsByUserIDFromCache() {
	key := userID_games_key_prefix + userID
	expected := []string{"game01", "game02"}
	suite.mockCacher.On("GetStringProperties", mock.Anything, key).Return(userGames, true)

	result, err := suite.client.GetGameIDsByUserID(suite.ctx, userID)
	assert.NoError(suite.T(), err)
	require.Equal(suite.T(), expected, result)
}

func (suite *CacheTest) TestGetGameIDsByUserIDNotFromCache() {
	key := userID_games_key_prefix + userID
	expected := []string{"game01", "game02"}

	suite.mockCacher.On("CacheStringProperties", mock.Anything, key, userGames).Return(nil)
	suite.mockClient.On("GetGameIDsByUserID", suite.ctx, userID).Return(expected, nil)
	suite.mockCacher.On("GetStringProperties", mock.Anything, key).Return(mock.Anything, false)

	result, err := suite.client.GetGameIDsByUserID(suite.ctx, userID)
	assert.NoError(suite.T(), err)
	require.Equal(suite.T(), expected, result)
}

func TestUtilsSuite(t *testing.T) {
	suite.Run(t, new(CacheTest))
}
