package rbac

import (
	"errors"
	"fmt"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"code.justin.tv/insights/piper-service/models"
	"github.com/afex/hystrix-go/hystrix"
	"github.com/twitchtv/twirp"
	"golang.org/x/net/context"
)

func init() {
	hystrix.Configure(map[string]hystrix.CommandConfig{
		"rbac_validate": {
			Timeout:               500,
			MaxConcurrentRequests: 10000,
		},
	})
}

// ErrUnknownResource signifies RBAC doesn't know about the resource
// and other authorization means should be used.
var ErrUnknownResource = errors.New("resource not known to rbac")

// Calls RBAC with the given validate query. Returns:
// - nil: user has access
// - ErrInternalError: internal error. if an RBAC response is required in the auth chain, surface this to user.
// - ErrUnknownResource: rbac doesn't know about resource and cannot make an authorization decision
// - ErrAccessForbidden: rbac knows about resource and user does not have access to it.
func (c *clientImpl) Validate(ctx context.Context, params *rbacrpc.ValidateQuery) error {
	return hystrix.Do("rbac_validate", func() error {
		validateResponse, err := c.rbac.ValidateByTwitchID(ctx, params)

		if err != nil {
			if twerr, ok := err.(twirp.Error); ok {
				if twerr.Code() == twirp.NotFound && twerr.Msg() == "resource" {
					// rbac doesn't know about it
					return ErrUnknownResource
				}
				// if we get any other error, it's unexpected.
				logx.Error(ctx, fmt.Sprintf("rbac.Validate error: %s", twerr), logx.Fields{
					"user_id":       params.UserId,
					"resource_id":   params.ResourceId,
					"resource_type": params.ResourceType,
					"permission":    params.Permission,
				})
			}
			return errx.New(models.ErrInternalError)
		}
		if !validateResponse.GetValid() {
			return models.ErrAccessForbidden
		}
		return nil
	}, nil)
}
