package s3

import (
	"path/filepath"
	"strconv"
	"time"

	"os"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/request"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"
)

const (
	s3PathDelimiter = "/"
)

//go:generate mockery -name Client
type Client interface {
	PreSignURL(bucket string, key string, filename string, urlExpireMinutes int) (string, error)
	UploadFileToS3(bucket, path, fileToUpload string) (string, error)
}

type clientImpl struct {
	client *s3.S3
}

func New(sess *session.Session) *clientImpl {
	return &clientImpl{
		client: s3.New(sess),
	}
}

func (s *clientImpl) PreSignURL(bucket string, key string, contentDispositionFileName string, urlExpireMinutes int) (string, error) {
	req := s.getRequest(bucket, key, contentDispositionFileName)
	ttl := getTTLInMinute(urlExpireMinutes)

	url, err := req.Presign(ttl)

	if err != nil {
		return "", err
	}
	return url, nil
}

func (s *clientImpl) getRequest(bucket string, key string, contentDispositionFileName string) *request.Request {
	req, _ := s.client.GetObjectRequest(&s3.GetObjectInput{
		Bucket:                     aws.String(bucket),
		Key:                        aws.String(key),
		ResponseContentDisposition: aws.String("attachment;filename=" + strconv.Quote(contentDispositionFileName)),
	})

	return req
}

func getTTLInMinute(ttl int) time.Duration {
	return time.Duration(ttl) * time.Minute
}

func (s *clientImpl) UploadFileToS3(bucket, path, fileToUpload string) (string, error) {
	key := path + filepath.Base(fileToUpload)

	file, err := os.Open(fileToUpload)
	if err != nil {
		return "", err
	}
	defer func() {
		cerr := file.Close()
		if err == nil {
			err = cerr
		}
	}()

	input := &s3.PutObjectInput{
		Bucket: aws.String(bucket),
		Key:    aws.String(key),
		Body:   file,
	}

	_, err = s.client.PutObject(input)
	if err != nil {
		return "", err
	}

	return filepath.Base(fileToUpload), nil
}
