package uploader

import (
	"fmt"

	"code.justin.tv/insights/piper-service/internal/config"
	"code.justin.tv/web/upload-service/rpc/uploader"
	"golang.org/x/net/context"
)

const (
	globalReadAccessURI = "uri=http://acs.amazonaws.com/groups/global/AllUsers"
	defaultImageType    = "jpg"
	minWidth            = 600
	minHeight           = 800
	maxSize             = "1048576B"
)

func (u *uploaderImpl) CreateUploadServiceRequest(ctx context.Context, gameID, gameName string, conf config.PiperConfig) (*uploader.UploadResponse, error) {
	req := &uploader.UploadRequest{
		Outputs: []*uploader.Output{
			{
				Transformations: []*uploader.Transformation{
					{
						Transformation: &uploader.Transformation_Resize{
							Resize: &uploader.Resize{
								Size: &uploader.Resize_Dimensions{
									Dimensions: &uploader.Dimensions{
										Width:  uint32(minWidth),
										Height: uint32(minHeight),
									},
								},
							},
						},
					},
				},
				Name: fmt.Sprintf("%s.%s", gameName, defaultImageType),
				Permissions: &uploader.Permissions{
					GrantRead: globalReadAccessURI,
				},
			},
		},
		PreValidation: &uploader.Validation{
			FileSizeLessThan: maxSize,
			Format:           "image",
			WidthConstraints: []*uploader.Constraint{
				{
					Value: float64(minWidth),
					Test:  ">=",
				},
			},
			HeightConstraints: []*uploader.Constraint{
				{
					Value: float64(minHeight),
					Test:  ">=",
				},
			},
		},
		Callback: &uploader.Callback{
			SnsTopicArn: conf.UploadCallbackSNSArn,
			Data:        []byte("{\"game_id\": \"" + gameID + "\", \"game_name\": \"" + gameName + "\"}"),
			PubsubTopic: "default",
		},
		OutputPrefix: "s3://" + conf.UploadBucket + "/",
		Monitoring: &uploader.Monitoring{
			RollbarToken: conf.RollbarApiToken,
		},
	}

	resp, err := u.uploaderClient.Create(ctx, req)
	if err != nil {
		return nil, err
	}
	return resp, nil
}
