package config

import (
	"flag"
	"time"

	"code.justin.tv/common/config"
)

type PiperConfig struct {
	ReportBucket                  string //ReportBucket is the bucket where reports are stored
	ReportPrefix                  string //ReportPrefix is s3 report prefix
	ExtensionReportPrefix         string
	DiscoveryHost                 string
	UsersServiceHost              string
	OwlHost                       string
	RollbarApiToken               string
	PiperDB                       *PiperDBConn
	DynamicReportPrefix           string
	SandStormRoleARN              string
	S2SService                    string
	AWSRegion                     string
	RedisClusterAddr              string
	UploadURL                     string
	UploadCallbackSNSArn          string
	UploadBucket                  string
	RBACHost                      string
	EMSHost                       string
	ModsEleriumHost               string
	ModsEleriumApiKey             string
	ModsEleriumApiTokenSecretPath string
	ModsEleriumApiToken           string
}

type PiperDBConn struct {
	PiperDBReaderHost     string
	PiperDBReaderPort     string
	PiperDBReaderUser     string
	PiperDBReaderPassword string
	PiperDBReaderDB       string
	PiperDBDriver         string
	PiperDBSecretPath     string
}

// SetupPiperConfig reads beanstalk configuration file
func SetupPiperConfig() (*PiperConfig, error) {
	config.Register(map[string]string{
		"report-bucket":                      "twitch-piper-reports-dev",
		"report-prefix":                      "games/",
		"extension-report-prefix":            "extensions/",
		"rollbar-api-token":                  "5603187275da49b0a6e3b0bc18e72e8c",
		"discovery-host":                     "http://staging.oldcategorydb.twitch.a2z.com",
		"users-service-host":                 "https://dev.users-service.twitch.a2z.com",
		"owl-host":                           "https://prod.owl.twitch.a2z.com",
		"piper-db-reader-host":               "piper-daily-incremental.cluster-c6v8b6dphigg.us-west-2.rds.amazonaws.com",
		"piper-db-reader-port":               "5432",
		"piper-db-reader-db":                 "daily_incremental",
		"piper-db-reader-user":               "incremental_02",
		"piper-db-reader-password":           "changeme",
		"piper-db-driver":                    "postgres",
		"dynamic-report-prefix":              "dynamic/",
		"sandstorm-role-arn":                 "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/sandstorm-agent-piper-service-staging",
		"s2s-service":                        "piper-service-staging",
		"piper-db-secret-path":               "insights/piper-daily-incremental/staging/incremental_02",
		"redis-cluster-addr":                 "piper-service-cache.eo6lva.clustercfg.usw2.cache.amazonaws.com:6379",
		"upload-url":                         "https://staging.upload-service.twitch.a2z.com",
		"upload-callback-sns-arn":            "arn:aws:sns:us-west-2:895328872748:games_image-callback",
		"upload-bucket":                      "twitch-piper-games-images-staging",
		"ems-host":                           "https://prod.ems.extensions.twitch.a2z.com",
		"rbac-host":                          "https://prod.devsite-rbac.twitch.a2z.com",
		"mods-elerium-host":                  "https://elerium.curseforge.com",
		"mods-elerium-api-key":               "X-Api-Token",
		"mods-elerium-api-token-secret-path": "insights/piper-service/staging/mods-elerium-api-token",
		"mods-elerium-api-token":             "changeme",
	})
	if err := config.Parse(); err != nil {
		return nil, err
	}

	if err := flag.Set("aws-region", "us-west-2"); err != nil {
		return nil, err
	}

	piperDBCfg := &PiperDBConn{
		PiperDBReaderHost:     config.MustResolve("piper-db-reader-host"),
		PiperDBReaderPort:     config.MustResolve("piper-db-reader-port"),
		PiperDBReaderDB:       config.MustResolve("piper-db-reader-db"),
		PiperDBReaderUser:     config.MustResolve("piper-db-reader-user"),
		PiperDBReaderPassword: config.MustResolve("piper-db-reader-password"),
		PiperDBDriver:         config.MustResolve("piper-db-driver"),
		PiperDBSecretPath:     config.MustResolve("piper-db-secret-path"),
	}

	piperCfg := &PiperConfig{
		ReportBucket:                  config.MustResolve("report-bucket"),
		ReportPrefix:                  config.MustResolve("report-prefix"),
		ExtensionReportPrefix:         config.MustResolve("extension-report-prefix"),
		DiscoveryHost:                 config.MustResolve("discovery-host"),
		UsersServiceHost:              config.MustResolve("users-service-host"),
		OwlHost:                       config.MustResolve("owl-host"),
		RollbarApiToken:               config.MustResolve("rollbar-api-token"),
		PiperDB:                       piperDBCfg,
		DynamicReportPrefix:           config.MustResolve("dynamic-report-prefix"),
		SandStormRoleARN:              config.MustResolve("sandstorm-role-arn"),
		S2SService:                    config.MustResolve("s2s-service"),
		AWSRegion:                     config.AwsRegion(),
		RedisClusterAddr:              config.MustResolve("redis-cluster-addr"),
		UploadURL:                     config.MustResolve("upload-url"),
		UploadCallbackSNSArn:          config.MustResolve("upload-callback-sns-arn"),
		UploadBucket:                  config.MustResolve("upload-bucket"),
		EMSHost:                       config.MustResolve("ems-host"),
		RBACHost:                      config.MustResolve("rbac-host"),
		ModsEleriumHost:               config.MustResolve("mods-elerium-host"),
		ModsEleriumApiKey:             config.MustResolve("mods-elerium-api-key"),
		ModsEleriumApiTokenSecretPath: config.MustResolve("mods-elerium-api-token-secret-path"),
		ModsEleriumApiToken:           config.MustResolve("mods-elerium-api-token"),
	}

	return piperCfg, nil
}

type DBDefaultConfig struct {
	MaxOpenConns       int
	MaxIdleConns       int
	ConnQueueSize      int
	MaxQueueSize       int
	ConnAcquireTimeout time.Duration
	RequestTimeout     time.Duration
	MaxConnAge         time.Duration
}

func SetupPiperDBReader() *DBDefaultConfig {
	return &DBDefaultConfig{
		MaxOpenConns:       int(1000),
		MaxIdleConns:       int(1000),
		MaxQueueSize:       int(5000),
		ConnAcquireTimeout: 10 * time.Second,
		RequestTimeout:     20 * time.Second,
		MaxConnAge:         50 * time.Second,
	}
}

type RedisDefaultConfig struct {
	ConnectTimeout    int
	ReadTimeout       int
	WriteTimeout      int
	MaxConnections    int
	MonitorInterval   int
	StatSampleRate    float32
	DefaultExpiration int
}

func SetupRedisCluster() *RedisDefaultConfig {
	return &RedisDefaultConfig{
		ConnectTimeout:    int(2),
		ReadTimeout:       int(2),
		WriteTimeout:      int(2),
		MaxConnections:    int(1000),
		MonitorInterval:   int(5),
		StatSampleRate:    float32(0.1),
		DefaultExpiration: int(7200),
	}
}
