package reports

import (
	"strings"
	"time"

	"code.justin.tv/insights/piper-service/models"
)

const (
	shortDateFormat = "2006-01-02"
	// Devsite ReportType represents the report that should be used in the dev dashboard, curseforge panel, etc.
	DevsiteGamesReportType      = "overview_v2"
	DevsiteExtensionsReportType = "overview_v2"
	DevsiteModsReportType       = "overview_v1"
)

// whitelistedReports is the configuration for reports that should be exposed in the API.
var whitelistedReports = map[string]map[string]*ReportConf{
	"games": {
		"overview_v1": &ReportConf{Retention: 90},
		"overview_v2": &ReportConf{Retention: 365},
	},
	"extensions": {
		"overview_v1": &ReportConf{Retention: 90},
		"overview_v2": &ReportConf{RetentionSince: time.Date(2018, 1, 31, 0, 0, 0, 0, time.UTC)},
	},
	"mods": {
		"overview_v1": &ReportConf{Retention: 90},
	},
}

// whitelistedReportTypes should be updated when a new activeReport is added
var whitelistedReportTypes = map[string][]string{
	"games":      {"overview_v1", "overview_v2"},
	"extensions": {"overview_v1", "overview_v2"},
	"mods":       {"overview_v1"},
	"drops":      {"campaigns_overview", "campaigns_top_streamers", "campaigns_drops_overview"},
}

type ReportConf struct {
	Retention      int       // days ago for default start date since default end date
	RetentionSince time.Time // fixed startDate (e.g. extensions_overview_v2)
}

func GetWhitelistedReportTypesByDomain(domain string) []string {
	return whitelistedReportTypes[domain]
}

func getReportConf(domain string, reportType string) *ReportConf {
	domainReports := whitelistedReports[domain]
	if domainReports == nil {
		return nil
	}
	return domainReports[reportType]
}

// IsWhitelisted returns true if the exact domain and reportType are registered in whitelistedReports.
func IsWhitelisted(domain, reportType string) bool {
	return getReportConf(domain, reportType) != nil
}

func GetFileNameNoUUID(filename string) string {
	ext := ".csv"
	name := strings.TrimRight(filename, ext)
	elem := strings.Split(name, "_")
	return strings.Join(elem[:len(elem)-1], "_") + ext
}

// GetFittedDateRange is to replace GetReportDatesByType, GetReportDefaultDates and GetReportDateRange
// it ignores domain, report type and retention, and compares input with db states to get the most fitted date range
// input dates are user input values, it must follow RFC3339 and basic checks
// db dates should be pulled from latest_domain_report_meta_data table in piper db, which already count in report retentions
func GetFittedDateRange(inputStartDate, inputEndDate string, dbStartDate, dbEndDate time.Time) (string, string, error) {
	// if user has no input dates, use db date ranges
	if inputStartDate == "" && inputEndDate == "" {
		return dbStartDate.Format(shortDateFormat), dbEndDate.Format(shortDateFormat), nil
	}

	// if one of the input dates is left out return ErrInvalidDateRange
	if inputStartDate == "" || inputEndDate == "" {
		return "", "", models.ErrInvalidDateRange
	}

	// Helix Endpoint requires timestamp to be RFC 3339 format, "2018-02-05T08:00:00Z"
	inputStart, err := time.Parse(time.RFC3339, inputStartDate)
	if err != nil {
		return "", "", models.ErrInvalidDateFormat
	}

	inputEnd, err := time.Parse(time.RFC3339, inputEndDate)
	if err != nil {
		return "", "", models.ErrInvalidDateFormat
	}

	if inputStart.After(inputEnd) {
		return "", "", models.ErrInvalidDateRange
	}

	if inputEnd.Before(dbStartDate) || inputStart.After(dbEndDate) {
		return "", "", models.ErrInvalidDateRange
	}

	if inputStart.Before(dbStartDate) {
		inputStart = dbStartDate
	}

	if inputEnd.After(dbEndDate) {
		inputEnd = dbEndDate
	}

	return inputStart.Format(shortDateFormat), inputEnd.Format(shortDateFormat), nil
}
