package reports_test

import (
	"reflect"
	"testing"
	"time"

	"code.justin.tv/insights/piper-service/lib/reports"
	"github.com/stretchr/testify/require"
)

func Test_IsWhitelisted(t *testing.T) {
	testCases := []struct {
		expected   bool
		domain     string
		reportType string
	}{
		{true, "games", "overview_v1"},
		{true, "games", "overview_v2"},
		{true, "extensions", "overview_v1"},

		{false, "games", "overview"}, // only registered types count as valid
		{false, "games", "nonexisting_foobar"},
		{false, "extensions", "nonexisting_foobar"},
		{false, "nonexistingdomain", "overview_v1"},
		{false, "", "overview_v1"},
		{false, "", ""},
	}
	for _, c := range testCases {
		actual := reports.IsWhitelisted(c.domain, c.reportType)
		if actual != c.expected {
			t.Errorf("IsWhitelisted domain: %q, reportType: %q, unexpected result: %v", c.domain, c.reportType, actual)
		}
	}
}

func Test_GetWhitelistedReportTypesByDomain(t *testing.T) {
	testCases := []struct {
		domain   string
		expected []string
	}{
		{"games", []string{"overview_v1", "overview_v2"}},
		{"foobar", nil},
		{"", nil},
	}
	for _, c := range testCases {
		actual := reports.GetWhitelistedReportTypesByDomain(c.domain)
		if !reflect.DeepEqual(actual, c.expected) {
			t.Errorf("GetWhitelistedReportTypesByDomain domain: %q, unexpected result: %v", c.domain, actual)
		}
	}
}

func Test_GetFittedDateRange(t *testing.T) {
	dbStart := time.Date(2018, 1, 1, 0, 0, 0, 0, time.UTC)
	dbEnd := time.Date(2019, 1, 1, 0, 0, 0, 0, time.UTC)

	testCasesWithError := []struct {
		inputStart string
		inputEnd   string
	}{
		{inputStart: "2018-01-01T00:00:00Z", inputEnd: ""},                     // only one of the input is empty
		{inputStart: "", inputEnd: "2018-01-01T00:00:00Z"},                     // only one of the input is empty
		{inputStart: "2018-01-01 00:00:00", inputEnd: "2018-03-01T00:00:00Z"},  // wrong date format
		{inputStart: "2018-03-01T00:00:00Z", inputEnd: "2018-07-11 00:00:00"},  // wrong date format
		{inputStart: "2018-03-01T00:00:00Z", inputEnd: "2018-01-01T00:00:00Z"}, // start is later than end
		{inputStart: "2017-01-01T00:00:00Z", inputEnd: "2017-03-01T00:00:00Z"}, // input end before dbstart
		{inputStart: "2019-01-02T00:00:00Z", inputEnd: "2019-03-01T00:00:00Z"}, // input start after dbend
	}

	for _, c := range testCasesWithError {
		t.Run("", func(t *testing.T) {
			rsDate, reDate, err := reports.GetFittedDateRange(c.inputStart, c.inputEnd, dbStart, dbEnd)
			require.Equal(t, rsDate, "", "returned start should be empty")
			require.Equal(t, reDate, "", "returned end should be empty")
			require.Error(t, err, "should be error")
		})
	}

	testCasesWillSucceed := []struct {
		inputStart string
		inputEnd   string
	}{
		{inputStart: "2018-01-02T00:00:00Z", inputEnd: "2018-07-01T00:00:00Z"}, // late input start and early input end
		{inputStart: "2017-01-02T00:00:00Z", inputEnd: "2018-07-01T00:00:00Z"}, // early input start and early input end
		{inputStart: "2018-01-02T00:00:00Z", inputEnd: "2019-07-01T00:00:00Z"}, // late input start and late input end
		{inputStart: "2017-01-02T00:00:00Z", inputEnd: "2019-07-01T00:00:00Z"}, // early input start and late input end
	}

	for _, c := range testCasesWillSucceed {
		t.Run("", func(t *testing.T) {
			rsDate, reDate, err := reports.GetFittedDateRange(c.inputStart, c.inputEnd, dbStart, dbEnd)
			require.NotEqual(t, rsDate, "", "returned start should not be empty")
			require.NotEqual(t, reDate, "", "returned end should not be empty")
			require.NoError(t, err, "no error")
		})
	}
}
