package main

import (
	"fmt"
	"net/http"
	"time"

	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	"code.justin.tv/insights/piper-service/internal/metrics"
	"code.justin.tv/insights/piper-service/internal/utils"

	"code.justin.tv/sse/malachai/pkg/events"
	"code.justin.tv/sse/malachai/pkg/s2s/callee"

	"code.justin.tv/insights/piper-service/internal/clients/elerium"

	"code.justin.tv/insights/piper-service/backend/mods"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/common/spade-client-go/spade"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/foundation/twitchserver"
	"code.justin.tv/insights/piper-service/api"
	"code.justin.tv/insights/piper-service/backend"
	"code.justin.tv/insights/piper-service/backend/drops"
	"code.justin.tv/insights/piper-service/backend/extensions"
	"code.justin.tv/insights/piper-service/backend/games"
	"code.justin.tv/insights/piper-service/backend/one_pager"
	"code.justin.tv/insights/piper-service/backend/s3report"
	"code.justin.tv/insights/piper-service/internal/clients/cache/elasticache"
	"code.justin.tv/insights/piper-service/internal/clients/discovery"
	"code.justin.tv/insights/piper-service/internal/clients/ems"
	"code.justin.tv/insights/piper-service/internal/clients/owl"
	"code.justin.tv/insights/piper-service/internal/clients/piperdb"
	"code.justin.tv/insights/piper-service/internal/clients/rbac"
	"code.justin.tv/insights/piper-service/internal/clients/s3"
	"code.justin.tv/insights/piper-service/internal/clients/sandstorm"
	"code.justin.tv/insights/piper-service/internal/clients/uploader"
	"code.justin.tv/insights/piper-service/internal/clients/users"
	piperconfig "code.justin.tv/insights/piper-service/internal/config"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/pkg/errors"
	"github.com/sirupsen/logrus"
	"golang.org/x/net/context"
)

func main() {
	ctx := context.Background()

	piperCfg, err := piperconfig.SetupPiperConfig()
	if err != nil {
		logx.Fatal(ctx, err)
	}

	fmt.Println("piper set up config")
	if piperCfg.RollbarApiToken != "" {
		logx.InitDefaultLogger(logx.Config{
			RollbarToken: piperCfg.RollbarApiToken,
			RollbarEnv:   utils.Environment(),
			RollbarLevels: []logrus.Level{
				logrus.PanicLevel,
				logrus.FatalLevel,
				logrus.ErrorLevel,
				logrus.WarnLevel,
			},
		})
	}

	logrus.StandardLogger().SetNoLock()

	defer logx.Wait()

	err = metrics.StartMetrics()
	if err != nil {
		logx.Fatal(ctx, err)
	}

	defer func() {
		if err := metrics.GetTelemetryStatter().Close(); err != nil {
			logx.Error(ctx, err)
		}
	}()

	s3client := configS3(piperCfg)

	fmt.Println("piper set up s3 client")

	redisCfg := piperconfig.SetupRedisCluster()

	cache, err := configCache(*piperCfg, *redisCfg)

	if err != nil {
		panic(fmt.Sprintf("failed to config cacher backend %v\n", err))
	}
	fmt.Println("piper set up cache")

	fmt.Println("piper set up sandstorm manager")
	sandstorm := configSandStorm(*piperCfg)

	spade, err := configureSpade()
	if err != nil {
		logx.Fatal(ctx, fmt.Sprintf("failed to config spade client %v\n", err))
	}
	fmt.Println("piper set up spade client")

	catDBHost := "https://us-west-2.prod.twitchcategorydb.s.twitch.a2z.com" // hack before removal, we only have prod environment
	discovery := configDiscovery(catDBHost, cache)

	if err != nil {
		logx.Fatal(ctx, fmt.Sprintf("failed to config discovery client %v\n", err))
	}

	users, err := configUserService(piperCfg.UsersServiceHost)

	if err != nil {
		logx.Fatal(ctx, fmt.Sprintf("failed to config users service client %v\n", err))
	}

	fmt.Println("piper set up user client")

	owlClient, err := configOwl(piperCfg.OwlHost, cache)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprintf("failed to config owl client %v\n", err))
	}

	fmt.Println("piper set up owl client")

	s3report, err := configS3Report(s3client)

	if err != nil {
		logx.Fatal(ctx, fmt.Sprintf("failed to config s3 report backend %v\n", err))
	}

	fmt.Println("piper set up Piper DB reader")

	dbCfg := piperconfig.SetupPiperDBReader()
	if err != nil {
		logx.Fatal(ctx, fmt.Sprintf("failed to config piper DB reader %v\n", err))
	}

	secret, err := sandstorm.GetSecret(ctx, piperCfg.PiperDB.PiperDBSecretPath)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprintf("failed to fetch piper db secret from sandstorm %v\n", err))
	}
	piperCfg.PiperDB.PiperDBReaderPassword = string(secret.Plaintext)

	piperdbclient, err := configPiperDB(*piperCfg.PiperDB, *dbCfg)

	fmt.Println("piper set up upload client")
	uploader := configUploadServiceClient(*piperCfg)

	fmt.Println("set up rbac client")
	rbac := configureRBAC(piperCfg.RBACHost, cache)

	fmt.Println("set up ems client")
	ems, err := configureEMS(piperCfg.EMSHost)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprintf("failed to config EMS %v\n", err))
	}

	secret, err = sandstorm.GetSecret(ctx, piperCfg.ModsEleriumApiTokenSecretPath)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprintf("failed to get mods elerium api token from sandstorm %v\n", err))
	}
	piperCfg.ModsEleriumApiToken = string(secret.Plaintext)

	fmt.Println("set up elerium client")
	elerium, err := configEleriumClient(piperCfg.ModsEleriumHost, cache)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprintf("failed to config elerium client %v\n", err))
	}

	gb, err := configGameBackend(s3client, cache, discovery, spade, users, s3report, piperdbclient, uploader, rbac)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprintf("failed to config games backend %v\n", err))
	}

	eb, err := configExtensionBackend(s3client, cache, spade, users, owlClient, s3report, piperdbclient, rbac)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprintf("failed to config extension backend %v\n", err))
	}

	db, err := configDropBackend(s3client, cache, spade, users, s3report, piperdbclient, rbac)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprintf("failed to config drops backend %v\n", err))
	}

	op, err := configureOnePagerBackend(piperdbclient, discovery, spade, rbac, users, ems, s3client, s3report, cache)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprintf("failed to config one pager backend %v\n", err))
	}

	mb, err := configureModBackend(piperdbclient, s3report, s3client, elerium, users, spade, cache)

	if err != nil {
		logx.Fatal(ctx, fmt.Sprintf("failed to config mods backend %v\n", err))
	}

	calleeClient := configureS2SClient(piperCfg)

	server, err := api.NewServer(gb, eb, db, op, mb, rbac, calleeClient, *piperCfg)
	if err != nil {
		logx.Fatal(ctx, err)
	}

	twitchserver.AddDefaultSignalHandlers()

	err = twitchserver.ListenAndServe(server, nil)
	if err != nil {
		logx.Fatal(ctx, err)

	}
}

func configCache(cfg piperconfig.PiperConfig, redisCfg piperconfig.RedisDefaultConfig) (backend.Cacher, error) {
	return elasticache.NewRedisClient(cfg, redisCfg)
}

func configS3Report(s3 s3.Client) (s3report.Backend, error) {
	return s3report.NewBackend(s3)
}

func configS3(conf *piperconfig.PiperConfig) s3.Client {
	return s3.New(session.New(&aws.Config{Region: &conf.AWSRegion}))
}

func configGameBackend(s3client s3.Client, backendCacher backend.Cacher, discovery discovery.Client, spade spade.Client, users users.Client, s3report s3report.Backend, piperdbclient piperdb.Client, uploadclient uploader.Client, rbacClient rbac.Client) (games.Backend, error) {
	return games.NewBackend(s3client, backendCacher, discovery, spade, users, s3report, piperdbclient, uploadclient, rbacClient)
}

func configExtensionBackend(s3client s3.Client, backendCacher backend.Cacher, spade spade.Client, users users.Client, owl owl.Client, s3report s3report.Backend, piperdbclient piperdb.Client, rbacclient rbac.Client) (extensions.Backend, error) {
	return extensions.NewBackend(s3client, backendCacher, spade, users, owl, s3report, piperdbclient, rbacclient)
}

func configDropBackend(s3client s3.Client, backendCacher backend.Cacher, spade spade.Client, users users.Client, s3report s3report.Backend, piperdbclient piperdb.Client, rbacclient rbac.Client) (drops.Backend, error) {
	return drops.NewBackend(s3client, spade, piperdbclient, rbacclient, users, s3report, backendCacher)
}

func configureOnePagerBackend(piperDBClient piperdb.Client, discoveryClient discovery.Client, spadeClient spade.Client, rbacClient rbac.Client, usersClient users.Client, emsClient ems.Client, s3client s3.Client, s3report s3report.Backend, backendCacher backend.Cacher) (one_pager.Backend, error) {
	return one_pager.NewBackend(piperDBClient, discoveryClient, spadeClient, rbacClient, usersClient, emsClient, s3client, s3report, backendCacher)
}

func configureModBackend(piperDBClient piperdb.Client, s3Report s3report.Backend, s3Client s3.Client, eleriumClient elerium.Client, usersClient users.Client, spadeClient spade.Client, cacher backend.Cacher) (mods.Backend, error) {
	return mods.NewBackend(piperDBClient, s3Report, s3Client, eleriumClient, usersClient, spadeClient, cacher)
}
func configDiscovery(host string, cache backend.Cacher) discovery.Client {
	conf := twitchclient.ClientConf{Host: host, Stats: metrics.GetTelemetryStatter()}
	return &discovery.CachedClient{
		Wrapped: discovery.NewDiscoveryClient(conf),
		Cacher:  cache,
	}
}

func configUserService(host string) (users.Client, error) {
	conf := twitchclient.ClientConf{Host: host, Stats: metrics.GetTelemetryStatter()}
	return users.NewUserClient(conf)
}

func configOwl(host string, cache backend.Cacher) (owl.Client, error) {
	conf := twitchclient.ClientConf{Host: host, Stats: metrics.GetTelemetryStatter()}
	owlClient, err := owl.NewOwlClient(conf)
	if err != nil {
		return nil, err
	}
	return &owl.CachedClient{
		Cacher:  cache,
		Wrapped: owlClient,
	}, nil
}

func configureSpade() (spade.Client, error) {
	httpClient := &http.Client{}
	return spade.NewClient(
		spade.InitHTTPClient(httpClient),
		spade.InitMaxConcurrency(1000),
		spade.InitStatHook(func(name string, httpStatus int, dur time.Duration) {
			metrics.Reporter().Report(fmt.Sprintf("spade.%s.%d", name, httpStatus), dur.Seconds(), telemetry.UnitSeconds)
		}),
	)
}

func configureRBAC(host string, cache backend.Cacher) rbac.Client {
	rbackClient := rbac.NewRBACClient(twitchclient.ClientConf{Host: host, Stats: metrics.GetTelemetryStatter()})
	return &rbac.CachedClient{
		Wrapped: rbackClient,
		Cacher:  cache,
	}
}

func configureEMS(host string) (ems.Client, error) {
	return ems.NewEMSClient(twitchclient.ClientConf{Host: host, Stats: metrics.GetTelemetryStatter()})
}

func configPiperDB(connCfg piperconfig.PiperDBConn, dbCfg piperconfig.DBDefaultConfig) (piperdb.Client, error) {
	return piperdb.NewPiperDB(connCfg, dbCfg)
}

func configSandStorm(conf piperconfig.PiperConfig) sandstorm.Client {
	return sandstorm.New(conf)
}

func configUploadServiceClient(conf piperconfig.PiperConfig) uploader.Client {
	return uploader.NewUploader(conf)
}

func configEleriumClient(host string, cacher backend.Cacher) (elerium.Client, error) {
	conf := twitchclient.ClientConf{Host: host, Stats: metrics.GetTelemetryStatter()}
	return elerium.NewEleriumClient(conf, cacher)
}

func configureS2SClient(conf *piperconfig.PiperConfig) callee.ClientAPI {
	if utils.Environment() == "development" {
		return nil
	}

	ctx := context.Background()

	eventWriter, err := events.NewEventLogger(events.Config{}, nil)
	if err != nil {
		logx.Error(ctx, errors.Wrap(err, "failed initialize s2s event logger"))
		return nil
	}

	calleeClient := &callee.Client{
		ServiceName:        conf.S2SService,
		EventsWriterClient: eventWriter,
		Config: &callee.Config{
			PassthroughMode: true,
		},
	}
	if err := calleeClient.Start(); err != nil {
		logx.Error(ctx, errors.Wrap(err, "failed to start s2s callee client"))
		return nil
	}

	fmt.Println("set up s2s callee client")

	return calleeClient
}
