resource "aws_iam_role" "app" {
  name = "${var.iam_role_name}"
  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "",
      "Effect": "Allow",
      "Principal": {
        "Service": "ec2.amazonaws.com"
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "app-sandstorm" {
  name = "${var.iam_role_name}-sandstorm"
  role = "${aws_iam_role.app.id}"
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Effect": "Allow",
      "Resource": "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/sandstorm-agent-piper-service-${var.environment}"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "app-s2s" {
  name = "${var.iam_role_name}-s2s"
  role = "${aws_iam_role.app.id}"
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [{
    "Effect": "Allow",
    "Action": "sts:AssumeRole",
    "Resource": "arn:aws:iam::180116294062:role/malachai/*"
  }]
}
EOF
}

resource "aws_iam_policy" "app-policy" {
  name = "${var.iam_role_name}-access-policy"
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": [
        "s3:Get*",
        "s3:List*",
        "s3:Put*"
      ],
      "Resource": [
          "arn:aws:s3:::twitch-piper-reports-dev",
          "arn:aws:s3:::twitch-piper-reports-dev/*",
          "arn:aws:s3:::twitch-piper-reports",
          "arn:aws:s3:::twitch-piper-reports/*"
      ],
      "Effect": "Allow"
    }
  ]
}
EOF
}

resource "aws_iam_policy_attachment" "app-role-policy-attach" {
    name = "{var.iam_role_name}-access-attachment"
    roles = ["${aws_iam_role.app.name}"]
    policy_arn = "${aws_iam_policy.app-policy.arn}"
}

data "aws_iam_policy" "AmazonEC2RoleforSSM" {
  arn = "arn:aws:iam::aws:policy/service-role/AmazonEC2RoleforSSM"
}

resource "aws_iam_role_policy_attachment" "ssm-attachment" {
  role = "${aws_iam_role.app.name}"
  policy_arn = "${data.aws_iam_policy.AmazonEC2RoleforSSM.arn}"
}

variable "grafana_production_role_arns" {
  type = "list"

  default = [
    # twitch-grafana-prod (grafana.xarth.tv)
    "arn:aws:iam::109561199089:role/grafana-production",

    # twitch-grafana-dev  (grafana-dev.twitch.a2z.com)
    "arn:aws:iam::963768028156:role/grafana-development",

    # twitch-central-aws  (deprecated: grafana.internal.justin.tv)
    "arn:aws:iam::007917851548:role/grafana-production",
  ]

  description = "roles used by grafana clusters to fetch the metrics from cloudwatch. This role is managed by #availability team"
}

variable "cloudwatch_read_only_policy_arn" {
  default     = "arn:aws:iam::aws:policy/CloudWatchReadOnlyAccess"
  description = "Policy with permissions required to get cloudwatch metrics in grafana"
}

data "aws_iam_policy_document" "grafana-cloudwatch-arp" {
  statement {
    effect = "Allow"

    actions = [
      "sts:AssumeRole",
    ]

    principals {
      type = "AWS"

      identifiers = "${var.grafana_production_role_arns}"
    }
  }
}

resource "aws_iam_role" "grafana-cloudwatch-read-only" {
  name               = "grafana-cloudwatch-read-only"
  assume_role_policy = "${data.aws_iam_policy_document.grafana-cloudwatch-arp.json}"
}

# https://grafana.com/docs/features/datasources/cloudwatch/#iam-policies
data "aws_iam_policy_document" "grafana-cloudwatch-access-policy" {
  statement {
    sid = "AllowReadingMetricsFromCloudWatch"

    actions = [
      "cloudwatch:DescribeAlarmsForMetric",
      "cloudwatch:ListMetrics",
      "cloudwatch:GetMetricStatistics",
      "cloudwatch:GetMetricData",
    ]

    resources = [
      "*",
    ]
  }

  statement {
    sid = "AllowReadingTagsInstancesRegionsFromEC2"

    actions = [
      "ec2:DescribeTags",
      "ec2:DescribeInstances",
      "ec2:DescribeRegions",
    ]

    resources = [
      "*",
    ]
  }

  statement {
    sid = "AllowReadingResourcesForTags"

    actions = [
      "tag:GetResources",
    ]

    resources = [
      "*",
    ]
  }
}

resource "aws_iam_policy" "grafana_policy" {
  name   = "grafana-cloudwatch-access-policy"
  path   = "/"
  policy = "${data.aws_iam_policy_document.grafana-cloudwatch-access-policy.json}"
}

resource "aws_iam_role_policy_attachment" "grafana_policy_attach" {
  role       = "${aws_iam_role.grafana-cloudwatch-read-only.name}"
  policy_arn = "${aws_iam_policy.grafana_policy.arn}"
}

output "role_arn" {
  value = "${aws_iam_role.grafana-cloudwatch-read-only.arn}"
}

output "role_name" {
  value = "${aws_iam_role.grafana-cloudwatch-read-only.name}"
}
