package tests

import (
	"testing"

	"fmt"

	"code.justin.tv/foundation/twitchclient"
	piperclient "code.justin.tv/insights/piper-service/client"
	"github.com/stretchr/testify/assert"
	"golang.org/x/net/context"
)

var (
	validUser     = "155668964"
	noGameUser    = "25410421"
	noCompanyUser = "145956020"
	notExistUser  = "0"
	domain        = "games"
	gameid        = "497425"
	reportType    = "overview_v2"
	instance      = "latest"
)

type TestGetGamesReports struct {
	PiperClient piperclient.Client
}

func (f *TestGetGamesReports) Test(t *testing.T) {
	runTests(t,
		testCase{"TestGetGamesReport", f.TestGetReportsValidUser},
		testCase{"TestGetGamesReportsNoCompanyUser", f.TestGetReportsNoCompanyUser},
		testCase{"TestGetGamesReportsNotExistUser", f.TestGetReportsNotExistUser},
		testCase{"TestGetGamesReportURL", f.TestGetReportURL},
		testCase{"TestGetGamesReportTypes", f.TestGetGamesReportTypes},
	)

	fmt.Println("Passed all tests in TestGetGamesReports.")
}

func (f *TestGetGamesReports) TestGetReportsValidUser(t *testing.T) {
	result, err := f.PiperClient.GetGamesReports(context.Background(), validUser, domain, &twitchclient.ReqOpts{})

	if err != nil {
		t.Fatal(t, "Failed TestGetReportsValidUser.", err)
	}

	assert.NotNil(t, result)
	assert.NotEqual(t, 0, len(result.Key))

	matched := false
	reportTypeTrue := ""

	for _, rp := range result.Key {
		if rp.GameID == gameid {
			matched = true
			reportTypeTrue = rp.ReportName
		}
	}
	assert.True(t, matched)
	assert.Equal(t, reportType, reportTypeTrue)

	fmt.Println("Passed TestGetReportsValidUser.")
}

func (f *TestGetGamesReports) TestGetReportsNoGameUser(t *testing.T) {
	_, err := f.PiperClient.GetGamesReports(context.Background(), noGameUser, domain, &twitchclient.ReqOpts{})

	if err == nil {
		t.Fatal(t, "Failed TestGetReportsNoGameUser.")
	}

	assert.Equal(t, err.Error(), "404: {\"status\":404,\"message\":\"User Not Map To Games\",\"error\":\"Not Found\"}")

	fmt.Println("Passed TestGetReportsNoGameUser.")
}

func (f *TestGetGamesReports) TestGetReportsNoCompanyUser(t *testing.T) {
	_, err := f.PiperClient.GetGamesReports(context.Background(), noCompanyUser, domain, &twitchclient.ReqOpts{})

	if err == nil {
		t.Fatal(t, "Failed TestGetReportsNoCompanyUser.")
	}

	assert.Equal(t, err.Error(), "403: User Has No Access To Any Games")

	fmt.Println("Passed TestGetReportsNoCompanyUser.")
}

func (f *TestGetGamesReports) TestGetReportsNotExistUser(t *testing.T) {
	_, err := f.PiperClient.GetGamesReports(context.Background(), notExistUser, domain, &twitchclient.ReqOpts{})

	if err == nil {
		t.Fatal(t, "Failed TestGetReportsNotExistUser.")
	}

	assert.Equal(t, err.Error(), "403: User Has No Access To Any Games")

	fmt.Println("Passed TestGetReportsNotExistUser.")
}

func (f *TestGetGamesReports) TestGetReportURL(t *testing.T) {
	signedurl, err := f.PiperClient.GetReportURL(context.Background(), validUser, domain, gameid, reportType, instance, "", &twitchclient.ReqOpts{})

	if err != nil {
		t.Fatal(t, "Failed TestGetReportURL.", err)
	}

	checkPresignedURL(t, signedurl)

	fmt.Println("Passed TestGetReportURL.")
}

func (f *TestGetGamesReports) TestGetGamesReportTypes(t *testing.T) {
	result, err := f.PiperClient.GetGamesReportTypes(context.Background(), validUser, domain, gameid, &twitchclient.ReqOpts{})

	if err != nil {
		t.Fatal(t, "Failed TestGetGamesReportTypes.", err)
	}

	assert.NotNil(t, result)
	assert.Equal(t, reportType, result.Key.ReportTypes[0])

	fmt.Println("Passed TestGetGamesReportTypes.")
}
