package tests

import (
	"fmt"
	"net/url"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"

	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/insights/piper-service/models"
)

func checkError(t *testing.T, expected models.CodedError, actual error, msgAndArgs ...interface{}) {
	require.Error(t, actual, msgAndArgs...)
	clientErr, ok := actual.(*twitchclient.Error)
	typeAssertionMsg := "error is not type twitchclient.Error: %v"
	typeAssertionArgs := []interface{}{actual}
	if len(msgAndArgs) > 0 {
		typeAssertionMsg = fmt.Sprintf("%s: ", msgAndArgs[0]) + typeAssertionMsg
	}
	if len(msgAndArgs) > 1 {
		typeAssertionArgs = append(msgAndArgs[1:], actual)
	}
	require.True(t, ok, "error is not type twitchclient.Error: %v", typeAssertionArgs)
	require.Equal(t, expected.StatusCode(), clientErr.StatusCode, msgAndArgs...)
	require.Equal(t, expected.Error(), clientErr.Message, msgAndArgs...)
}

func checkPresignedURL(t *testing.T, signedurl string) {
	u, err := url.Parse(signedurl)
	require.NoError(t, err)
	assert.Equal(t, u.Scheme, "https")

	// s3 wills one of the 2 urls:
	// twitch-piper-reports-dev.s3-us-west-2.amazonaws.com
	// twitch-piper-reports-dev.s3.us-west-2.amazonaws.com
	assert.Contains(t, u.Host, "twitch-piper-reports-dev")
	assert.Contains(t, u.Host, "s3")
	assert.Contains(t, u.Host, "us-west-2")
	assert.Contains(t, u.Host, "amazonaws.com")

	m, err := url.ParseQuery(u.RawQuery)
	require.NoError(t, err)
	require.NotEmpty(t, m["X-Amz-Date"])
	assert.NotNil(t, m["X-Amz-Date"][0])
	require.NotEmpty(t, m["X-Amz-Algorithm"])
	assert.Equal(t, m["X-Amz-Algorithm"][0], "AWS4-HMAC-SHA256")
	require.NotEmpty(t, m["X-Amz-Credential"])
	assert.NotNil(t, m["X-Amz-Credential"][0])
	require.NotEmpty(t, m["X-Amz-SignedHeaders"])
	assert.Equal(t, m["X-Amz-SignedHeaders"][0], "host")
}

type testCase struct {
	Name     string
	TestFunc func(t *testing.T)
}

func runTests(t *testing.T, tests ...testCase) {
	for i, tc := range tests {
		t.Run(tc.Name, tc.TestFunc)
		if i != len(tests)-1 {
			time.Sleep(100 * time.Millisecond)
		}
	}
}
