package ems

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"

	"code.justin.tv/foundation/twitchclient"

	"code.justin.tv/gds/gds/extensions/ems/documents"
)

// DiscomanClient defines the (currently extended of EMS Client) Client interface
type DiscomanClient interface {
	// AddCategory creates a new category.  The requesting user must be whitelisted to edit categories.
	AddCategory(ctx context.Context, params documents.AddCategoryRequest, reqOpts *twitchclient.ReqOpts) (*documents.CategoryDocument, error)

	// AddExtensionToCategory adds an extension to the given category
	AddExtensionToCategory(ctx context.Context, eid, cid string, params documents.AddExtensionToCategoryRequest, reqOpts *twitchclient.ReqOpts) (*documents.ExtensionCategoryMembershipDocument, error)

	// AddDeveloperCategoryToExtension sets the category on the provided extension version.
	AddDeveloperCategoryToExtension(ctx context.Context, eid, version, cid string, reqOpts *twitchclient.ReqOpts) (*documents.ExtensionCategoryMembershipDocument, error)

	// AddGameToExtension sets the game on the provided extension version.
	AddGameToExtension(ctx context.Context, eid, version string, gid int, reqOpts *twitchclient.ReqOpts) (*documents.ExtensionGameMembershipDocument, error)

	// EditCategoryTranslation edits or creates a new localized name and description for a category
	EditCategoryTranslation(ctx context.Context, cid, language string, params documents.EditCategoryTranslationRequest, reqOpts *twitchclient.ReqOpts) (*documents.CategoryDocument, error)

	// GetCategories gets a list of categories.  It supports pagination.
	GetCategories(ctx context.Context, params documents.GetCategoriesRequest, reqOpts *twitchclient.ReqOpts) (*documents.CategoriesDocument, error)

	// GetCategory gets a category given a category id and language
	GetCategory(ctx context.Context, cid, language string, reqOpts *twitchclient.ReqOpts) (*documents.CategoryDocument, error)

	// GetCategoryExtensions gets a paginated list of extensions in the provided category ID
	GetCategoryExtensions(ctx context.Context, cid string, params documents.GetCategoryExtensionsRequest, reqOpts *twitchclient.ReqOpts) (*documents.CategoryExtensionsDocument, error)

	// DeleteCategory deletes a category.  It is an error to delete a developer category that is nonempty.
	DeleteCategory(ctx context.Context, cid string, reqOpts *twitchclient.ReqOpts) (*documents.CategoryDocument, error)

	// DeleteGameToExtension sets the game on the provided extension version.
	DeleteGameFromExtension(ctx context.Context, eid, version string, gid int, reqOpts *twitchclient.ReqOpts) (*documents.ExtensionGameMembershipDocument, error)

	// OrderCategories gives a bulk reordering endpoint for categories
	OrderCategories(ctx context.Context, params documents.OrderCategoriesRequest, reqOpts *twitchclient.ReqOpts) (*documents.CategoryOrderDocument, error)

	// OrderCategory gives a bulk reordering endpoint for the entries in a single category
	OrderCategory(ctx context.Context, cid string, params documents.OrderCategoryRequest, reqOpts *twitchclient.ReqOpts) (*documents.CategoryDocument, error)

	// RemoveExtensionFromCategory removes an extension from the given category
	RemoveExtensionFromCategory(ctx context.Context, eid, cid string, reqOpts *twitchclient.ReqOpts) (*documents.ExtensionCategoryMembershipDocument, error)

	// UpdateCategory updates an existing category.  The requesting user must be whitelisted to edit categories.
	UpdateCategory(ctx context.Context, cid string, params documents.AddCategoryRequest, reqOpts *twitchclient.ReqOpts) (*documents.CategoryDocument, error)

	// CreateFeaturedSchedule adds a new featured carousel schedule.  The requesting user must be whitelisted to edit categories.
	CreateFeaturedSchedule(ctx context.Context, doc documents.AddFeaturedScheduleRequest, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedScheduleDocument, error)

	// GetFeaturedSchedules returns a paginated list of featured carousel schedules.
	GetFeaturedSchedules(ctx context.Context, offset, limit int, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedSchedulesDocument, error)

	// GetFeaturedSchedule returns a single specified featured carousel schedule.
	GetFeaturedSchedule(ctx context.Context, fsid string, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedScheduleDocument, error)

	// DeleteFeaturedSchedule deletes the specified featured carousel schedule.  The requesting user must be whitelisted to edit categories.
	DeleteFeaturedSchedule(ctx context.Context, fsid string, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedScheduleDocument, error)

	// UpdateFeaturedSchedule updates the specified featured carousel schedule.  The requesting user must be whitelisted to edit categories.
	UpdateFeaturedSchedule(ctx context.Context, fsid string, doc documents.AddFeaturedScheduleRequest, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedScheduleDocument, error)

	// CreateFeaturedCarousel adds a new featured carousel schedule.  The requesting user must be whitelisted to edit categories.
	CreateFeaturedCarousel(ctx context.Context, doc documents.AddFeaturedCarouselRequest, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedCarouselDocument, error)

	// GetFeaturedCarousels returns a paginated list of featured carousels. The requesting user must be whitelisted to edit categories.
	GetFeaturedCarousels(ctx context.Context, offset, limit int, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedCarouselsDocument, error)

	// GetFeaturedCarousel returns a single specified featured carousel schedule.
	GetFeaturedCarousel(ctx context.Context, fcid string, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedCarouselDocument, error)

	// DeleteFeaturedCarousel deletes the specified featured carousel.  The requesting user must be whitelisted to edit categories.
	DeleteFeaturedCarousel(ctx context.Context, fcid string, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedCarouselDocument, error)

	// UpdateFeaturedCarousel updates the specified featured carousel.  The requesting user must be whitelisted to edit categories.
	UpdateFeaturedCarousel(ctx context.Context, fcid string, doc documents.AddFeaturedCarouselRequest, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedCarouselDocument, error)

	// OrderFeaturedCarousel updates the ordering of the entries in the specified featured carousel.  The requesting user must be whitelisted to edit categories.
	OrderFeaturedCarousel(ctx context.Context, fcid string, doc documents.OrderFeaturedCarouselRequest, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedCarouselDocument, error)

	// CreateFeaturedCarouselEntry adds a new featured carousel entry.  The requesting user must be whitelisted to edit categories.
	CreateFeaturedCarouselEntry(ctx context.Context, doc documents.AddFeaturedCarouselEntryRequest, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedCarouselEntryDocument, error)

	// GetFeaturedCarouselEntries returns a paginated list of featured carousel entries. The requesting user must be whitelisted to edit categories.
	GetFeaturedCarouselEntries(ctx context.Context, offset, limit int, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedCarouselEntriesDocument, error)

	// GetFeaturedCarouselEntry returns a single specified featured carousel entry.
	GetFeaturedCarouselEntry(ctx context.Context, fceid string, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedCarouselEntryDocument, error)

	// DeleteFeaturedCarouselEntry deletes the specified featured carousel entry.  The requesting user must be whitelisted to edit categories.
	DeleteFeaturedCarouselEntry(ctx context.Context, fceid string, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedCarouselEntryDocument, error)

	// UpdateFeaturedCarouselEntry updates the specified featured carousel entry.  The requesting user must be whitelisted to edit categories.
	UpdateFeaturedCarouselEntry(ctx context.Context, fceid string, doc documents.AddFeaturedCarouselEntryRequest, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedCarouselEntryDocument, error)
}

func (c *clientImpl) AddCategory(ctx context.Context, params documents.AddCategoryRequest, reqOpts *twitchclient.ReqOpts) (*documents.CategoryDocument, error) {
	u := url.URL{Path: "/categories"}

	bodyBytes, err := json.Marshal(params)
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest(http.MethodPost, u.String(), bytes.NewReader(bodyBytes))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.add_category",
		StatSampleRate: defaultStatSampleRate,
	})
	var document documents.CategoryDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) AddExtensionToCategory(ctx context.Context, eid, cid string, params documents.AddExtensionToCategoryRequest, reqOpts *twitchclient.ReqOpts) (*documents.ExtensionCategoryMembershipDocument, error) {
	u := url.URL{Path: fmt.Sprintf("/categories/%s/extensions/%s", cid, eid)}

	bodyBytes, err := json.Marshal(params)
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest(http.MethodPost, u.String(), bytes.NewReader(bodyBytes))

	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.add_extension_to_category",
		StatSampleRate: defaultStatSampleRate,
	})

	var document documents.ExtensionCategoryMembershipDocument
	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) AddDeveloperCategoryToExtension(ctx context.Context, eid, version, cid string, reqOpts *twitchclient.ReqOpts) (*documents.ExtensionCategoryMembershipDocument, error) {
	u := url.URL{Path: fmt.Sprintf("/extensions/%s/%s/categories/%s", eid, version, cid)}

	req, err := c.NewRequest(http.MethodPost, u.String(), nil)

	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.add_dev_category_to_extension",
		StatSampleRate: defaultStatSampleRate,
	})

	var document documents.ExtensionCategoryMembershipDocument
	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) AddGameToExtension(ctx context.Context, eid, version string, gid int, reqOpts *twitchclient.ReqOpts) (*documents.ExtensionGameMembershipDocument, error) {
	u := url.URL{Path: fmt.Sprintf("/extensions/%s/%s/games/%d", eid, version, gid)}

	req, err := c.NewRequest(http.MethodPost, u.String(), nil)

	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.add_game_to_extension",
		StatSampleRate: defaultStatSampleRate,
	})

	var document documents.ExtensionGameMembershipDocument
	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) DeleteGameFromExtension(ctx context.Context, eid, version string, gid int, reqOpts *twitchclient.ReqOpts) (*documents.ExtensionGameMembershipDocument, error) {
	u := url.URL{Path: fmt.Sprintf("/extensions/%s/%s/games/%d", eid, version, gid)}

	req, err := c.NewRequest(http.MethodDelete, u.String(), nil)

	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.delete_game_from_extension",
		StatSampleRate: defaultStatSampleRate,
	})

	var document documents.ExtensionGameMembershipDocument
	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) DeleteCategory(ctx context.Context, cid string, reqOpts *twitchclient.ReqOpts) (*documents.CategoryDocument, error) {
	u := url.URL{Path: fmt.Sprintf("/categories/%s", cid)}

	req, err := c.NewRequest(http.MethodDelete, u.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.delete_category",
		StatSampleRate: defaultStatSampleRate,
	})

	var document documents.CategoryDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) EditCategoryTranslation(ctx context.Context, cid string, lang string, params documents.EditCategoryTranslationRequest, reqOpts *twitchclient.ReqOpts) (*documents.CategoryDocument, error) {
	u := url.URL{Path: fmt.Sprintf("/categories/%s/translations/%s", cid, lang)}

	bodyBytes, err := json.Marshal(params)
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest(http.MethodPut, u.String(), bytes.NewReader(bodyBytes))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.edit_category_translation",
		StatSampleRate: defaultStatSampleRate,
	})
	var document documents.CategoryDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) GetCategories(ctx context.Context, params documents.GetCategoriesRequest, reqOpts *twitchclient.ReqOpts) (*documents.CategoriesDocument, error) {
	u := url.URL{Path: "/categories"}
	v := url.Values{}
	v.Set("limit", fmt.Sprintf("%d", params.Limit))
	v.Set("offset", fmt.Sprintf("%d", params.Offset))
	if params.Type != "" {
		v.Set("type", params.Type)
	}
	if params.IncludeHidden {
		v.Set("include_hidden", "1")
	}
	if params.IncludeDeleted {
		v.Set("include_deleted", "1")
	}
	if params.Language != "" {
		v.Set("lang", params.Language)
	}
	u.RawQuery = v.Encode()

	req, err := c.NewRequest(http.MethodGet, u.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.get_categories",
		StatSampleRate: defaultStatSampleRate,
	})

	var document documents.CategoriesDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) GetCategory(ctx context.Context, cid, language string, reqOpts *twitchclient.ReqOpts) (*documents.CategoryDocument, error) {
	u := url.URL{Path: fmt.Sprintf("/categories/%s", cid)}
	v := url.Values{}
	if language != "" {
		v.Set("lang", language)
	}
	u.RawQuery = v.Encode()

	req, err := c.NewRequest(http.MethodGet, u.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.get_category",
		StatSampleRate: defaultStatSampleRate,
	})

	var document documents.CategoryDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) GetCategoryExtensions(ctx context.Context, cid string, params documents.GetCategoryExtensionsRequest, reqOpts *twitchclient.ReqOpts) (*documents.CategoryExtensionsDocument, error) {
	u := url.URL{Path: fmt.Sprintf("/categories/%s/extensions", cid)}
	values := url.Values{}
	values.Set("limit", fmt.Sprintf("%d", params.Limit))
	values.Set("offset", fmt.Sprintf("%d", params.Offset))
	u.RawQuery = values.Encode()

	req, err := c.NewRequest(http.MethodGet, u.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.get_category_extensions",
		StatSampleRate: defaultStatSampleRate,
	})

	var document documents.CategoryExtensionsDocument
	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) OrderCategories(ctx context.Context, params documents.OrderCategoriesRequest, reqOpts *twitchclient.ReqOpts) (*documents.CategoryOrderDocument, error) {
	url := url.URL{Path: "/categories/order"}

	bodyBytes, err := json.Marshal(params)
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest(http.MethodPost, url.String(), bytes.NewReader(bodyBytes))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.order_categories",
		StatSampleRate: defaultStatSampleRate,
	})

	var document documents.CategoryOrderDocument
	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) OrderCategory(ctx context.Context, cid string, params documents.OrderCategoryRequest, reqOpts *twitchclient.ReqOpts) (*documents.CategoryDocument, error) {
	url := url.URL{Path: fmt.Sprintf("/categories/%s/order", cid)}

	bodyBytes, err := json.Marshal(params)
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest(http.MethodPost, url.String(), bytes.NewReader(bodyBytes))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.order_category",
		StatSampleRate: defaultStatSampleRate,
	})
	var document documents.CategoryDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) RemoveExtensionFromCategory(ctx context.Context, eid, cid string, reqOpts *twitchclient.ReqOpts) (*documents.ExtensionCategoryMembershipDocument, error) {
	u := url.URL{Path: fmt.Sprintf("/categories/%s/extensions/%s", cid, eid)}

	req, err := c.NewRequest(http.MethodDelete, u.String(), nil)

	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.remove_extension_from_category",
		StatSampleRate: defaultStatSampleRate,
	})
	var document documents.ExtensionCategoryMembershipDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) UpdateCategory(ctx context.Context, cid string, params documents.AddCategoryRequest, reqOpts *twitchclient.ReqOpts) (*documents.CategoryDocument, error) {
	u := url.URL{Path: fmt.Sprintf("/categories/%s", cid)}

	bodyBytes, err := json.Marshal(params)
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest(http.MethodPut, u.String(), bytes.NewReader(bodyBytes))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.update_category",
		StatSampleRate: defaultStatSampleRate,
	})
	var document documents.CategoryDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) CreateFeaturedSchedule(ctx context.Context, params documents.AddFeaturedScheduleRequest, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedScheduleDocument, error) {
	u := url.URL{Path: "/featured_schedules"}

	bodyBytes, err := json.Marshal(params)
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest(http.MethodPost, u.String(), bytes.NewReader(bodyBytes))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.create_featured_schedule",
		StatSampleRate: defaultStatSampleRate,
	})
	var document documents.FeaturedScheduleDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) GetFeaturedSchedules(ctx context.Context, offset, limit int, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedSchedulesDocument, error) {
	u := url.URL{Path: "/featured_schedules"}
	v := url.Values{}
	v.Set("limit", fmt.Sprintf("%d", limit))
	v.Set("offset", fmt.Sprintf("%d", offset))
	u.RawQuery = v.Encode()

	req, err := c.NewRequest(http.MethodGet, u.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.get_featured_schedules",
		StatSampleRate: defaultStatSampleRate,
	})

	var document documents.FeaturedSchedulesDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) GetFeaturedSchedule(ctx context.Context, fsid string, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedScheduleDocument, error) {
	u := url.URL{Path: fmt.Sprintf("/featured_schedules/%s", fsid)}

	req, err := c.NewRequest(http.MethodGet, u.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.get_featured_schedule",
		StatSampleRate: defaultStatSampleRate,
	})

	var document documents.FeaturedScheduleDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) DeleteFeaturedSchedule(ctx context.Context, fsid string, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedScheduleDocument, error) {
	u := url.URL{Path: fmt.Sprintf("/featured_schedules/%s", fsid)}

	req, err := c.NewRequest(http.MethodDelete, u.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.delete_featured_schedule",
		StatSampleRate: defaultStatSampleRate,
	})

	var document documents.FeaturedScheduleDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) UpdateFeaturedSchedule(ctx context.Context, fsid string, params documents.AddFeaturedScheduleRequest, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedScheduleDocument, error) {
	u := url.URL{Path: fmt.Sprintf("/featured_schedules/%s", fsid)}

	bodyBytes, err := json.Marshal(params)
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest(http.MethodPut, u.String(), bytes.NewReader(bodyBytes))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.update_featured_schedule",
		StatSampleRate: defaultStatSampleRate,
	})
	var document documents.FeaturedScheduleDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) CreateFeaturedCarousel(ctx context.Context, params documents.AddFeaturedCarouselRequest, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedCarouselDocument, error) {
	u := url.URL{Path: "/featured_carousels"}

	bodyBytes, err := json.Marshal(params)
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest(http.MethodPost, u.String(), bytes.NewReader(bodyBytes))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.create_featured_carousel",
		StatSampleRate: defaultStatSampleRate,
	})
	var document documents.FeaturedCarouselDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) GetFeaturedCarousels(ctx context.Context, offset, limit int, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedCarouselsDocument, error) {
	u := url.URL{Path: "/featured_carousels"}
	v := url.Values{}
	v.Set("limit", fmt.Sprintf("%d", limit))
	v.Set("offset", fmt.Sprintf("%d", offset))
	u.RawQuery = v.Encode()

	req, err := c.NewRequest(http.MethodGet, u.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.get_featured_carousels",
		StatSampleRate: defaultStatSampleRate,
	})

	var document documents.FeaturedCarouselsDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) GetFeaturedCarousel(ctx context.Context, fcid string, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedCarouselDocument, error) {
	u := url.URL{Path: fmt.Sprintf("/featured_carousels/%s", fcid)}

	req, err := c.NewRequest(http.MethodGet, u.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.get_featured_carousel",
		StatSampleRate: defaultStatSampleRate,
	})

	var document documents.FeaturedCarouselDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) DeleteFeaturedCarousel(ctx context.Context, fcid string, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedCarouselDocument, error) {
	u := url.URL{Path: fmt.Sprintf("/featured_carousels/%s", fcid)}

	req, err := c.NewRequest(http.MethodDelete, u.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.delete_featured_carousel",
		StatSampleRate: defaultStatSampleRate,
	})

	var document documents.FeaturedCarouselDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) UpdateFeaturedCarousel(ctx context.Context, fcid string, params documents.AddFeaturedCarouselRequest, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedCarouselDocument, error) {
	u := url.URL{Path: fmt.Sprintf("/featured_carousels/%s", fcid)}

	bodyBytes, err := json.Marshal(params)
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest(http.MethodPut, u.String(), bytes.NewReader(bodyBytes))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.update_featured_carousel",
		StatSampleRate: defaultStatSampleRate,
	})
	var document documents.FeaturedCarouselDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) OrderFeaturedCarousel(ctx context.Context, fcid string, params documents.OrderFeaturedCarouselRequest, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedCarouselDocument, error) {
	u := url.URL{Path: fmt.Sprintf("/featured_carousels/%s/order", fcid)}

	bodyBytes, err := json.Marshal(params)
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest(http.MethodPost, u.String(), bytes.NewReader(bodyBytes))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.order_featured_carousel",
		StatSampleRate: defaultStatSampleRate,
	})
	var document documents.FeaturedCarouselDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) CreateFeaturedCarouselEntry(ctx context.Context, params documents.AddFeaturedCarouselEntryRequest, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedCarouselEntryDocument, error) {
	u := url.URL{Path: "/featured_carousel_entries"}

	bodyBytes, err := json.Marshal(params)
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest(http.MethodPost, u.String(), bytes.NewReader(bodyBytes))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.create_featured_carousel_entry",
		StatSampleRate: defaultStatSampleRate,
	})
	var document documents.FeaturedCarouselEntryDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) GetFeaturedCarouselEntries(ctx context.Context, offset, limit int, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedCarouselEntriesDocument, error) {
	u := url.URL{Path: "/featured_carousel_entries"}
	v := url.Values{}
	v.Set("limit", fmt.Sprintf("%d", limit))
	v.Set("offset", fmt.Sprintf("%d", offset))
	u.RawQuery = v.Encode()

	req, err := c.NewRequest(http.MethodGet, u.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.get_featured_carousel_entries",
		StatSampleRate: defaultStatSampleRate,
	})

	var document documents.FeaturedCarouselEntriesDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) GetFeaturedCarouselEntry(ctx context.Context, fceid string, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedCarouselEntryDocument, error) {
	u := url.URL{Path: fmt.Sprintf("/featured_carousel_entries/%s", fceid)}

	req, err := c.NewRequest(http.MethodGet, u.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.get_featured_carousel_entry",
		StatSampleRate: defaultStatSampleRate,
	})

	var document documents.FeaturedCarouselEntryDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) DeleteFeaturedCarouselEntry(ctx context.Context, fceid string, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedCarouselEntryDocument, error) {
	u := url.URL{Path: fmt.Sprintf("/featured_carousel_entries/%s", fceid)}

	req, err := c.NewRequest(http.MethodDelete, u.String(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.delete_featured_carousel_entry",
		StatSampleRate: defaultStatSampleRate,
	})

	var document documents.FeaturedCarouselEntryDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}

func (c *clientImpl) UpdateFeaturedCarouselEntry(ctx context.Context, fceid string, params documents.AddFeaturedCarouselEntryRequest, reqOpts *twitchclient.ReqOpts) (*documents.FeaturedCarouselEntryDocument, error) {
	u := url.URL{Path: fmt.Sprintf("/featured_carousel_entries/%s", fceid)}

	bodyBytes, err := json.Marshal(params)
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest(http.MethodPut, u.String(), bytes.NewReader(bodyBytes))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.extensions.update_featured_carousel_entry",
		StatSampleRate: defaultStatSampleRate,
	})
	var document documents.FeaturedCarouselEntryDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, nil
}
