package protocol

import (
	"net/http"

	"code.justin.tv/gds/gds/golibs/errors"
)

const (
	// ErrCodeExtensionNotInstalled is the error code for ErrExtensionNotInstalled
	ErrCodeExtensionNotInstalled = "extension_not_installed"
	// ErrExtensionIDAndVersionRequired
	ErrCodeExtensionIDAndVersionRequired = "extension_id_and_version_required"
	// ErrExtensionIDAndVersionRequired
	ErrCodeBadPagingToken = "bad_paging_token"
	// ErrExtensionIDAndVersionRequired
	ErrCodeAssetMismatch = "asset_mismatch"
	// ErrExtensionIDAndVersionRequired
	ErrCodeInvalidAssetHash = "invalid_asset_hash"
	// ErrCodeChannelNotMonetized
	ErrCodeChannelNotMonetized = "channel_not_monetized"
	// ErrCodeInvalidSubscriptionsSupportTransition when the subscriptions support level of an extension is
	// attempted to be changed when already once released with the level set to "required"
	ErrCodeInvalidSubscriptionsSupportTransition = "invalid_subscriptions_support_transition"
	// ErrCodeBitsIneligibleTransition is the error code for ErrBitsIneligibleTransition
	ErrCodeBitsIneligibleTransition = "forbidden_transition_to_bits_required"
	// ErrCodeNullUploader is the error code for ErrNullUploader
	ErrCodeNullUploader = "null_uploader"
	// ErrCodeInvalidImageAssetsType is the error code for ErrInvalidImageAssetsType
	ErrCodeInvalidImageAssetsType = "invalid_image_asset_type"
	// ErrCodeInvalidUploadImageBody is the error code for ErrInvalidUploadImageBody
	ErrCodeInvalidUploadImageBody = "invalid_upload_image_body"
	// ErrCodeNotEnoughUploadLogo is the error code for ErrNotEnoughUploadLogo
	ErrCodeNotEnoughUploadLogo = "not_enough_upload_logo"
	// ErrCodeNotEnoughUploadTaskbar is the error code for ErrNotEnoughUploadTaskbar
	ErrCodeNotEnoughUploadTaskbar = "not_enough_upload_taskbar"
	// ErrCodeNotEnoughUploadDiscovery is the error code for ErrNotEnoughUploadDiscovery
	ErrCodeNotEnoughUploadDiscovery = "not_enough_upload_discovery"
	// ErrCodeNotEnoughUploadScreenshot is the error code for ErrNotEnoughUploadScreenshot
	ErrCodeNotEnoughUploadScreenshot = "not_enough_upload_screenshot"
	// ErrCodeTooManyUploadLogo is the error code for ErrTooManyUploadLogo
	ErrCodeTooManyUploadLogo = "too_many_upload_logo"
	// ErrCodeTooManyUploadTaskbar is the error code for ErrTooManyUploadTaskbar
	ErrCodeTooManyUploadTaskbar = "too_many_upload_taskbar"
	// ErrCodeTooManyUploadDiscovery is the error code for ErrTooManyUploadDiscovery
	ErrCodeTooManyUploadDiscovery = "too_many_upload_discovery"
	// ErrCodeTooManyUploadScreenshot is the error code for ErrTooManyUploadScreenshot
	ErrCodeTooManyUploadScreenshot = "too_many_upload_screenshot"
	// ErrCodeAssetNotFoundManifest is the error code for ErrAssetNotFoundManifest
	ErrCodeAssetNotFoundManifest = "asset_not_found_in_manifest"
	// ErrCodeInvalidImageUrl is returned when the image url specified is invalid
	ErrCodeInvalidImageUrl = "invalid_image_url"
	// ErrCodeInvalidIconSize is returned when the icon size specified is invalid
	ErrCodeInvalidIconSize = "invalid_icon_size"
	// ErrCodeUsingRelativeWithAbsolutePaths is returned encountering an extension with
	// a mix of relative and absolute paths
	ErrCodeUsingRelativeWithAbsolutePaths = "using_relative_with_absolute_paths"
	// ErrCodeVersionAlreadyExists is returned when trying to clone to version that already exists
	ErrCodeVersionAlreadyExists = "version_already_exists"
)

var (
	// ErrExtensionNotInstalled is if the user attempts to activate or set feature flags on an
	// extension which was not already installed on their channel
	ErrExtensionNotInstalled = errors.NewBuilder("extension not installed").
					WithErrorCode(ErrCodeExtensionNotInstalled).
					WithHTTPStatus(http.StatusConflict).
					Build()

	ErrExtensionIDAndVersionRequired = errors.NewBuilder("extension id and version required").
						WithErrorCode(ErrCodeExtensionIDAndVersionRequired).
						WithHTTPStatus(http.StatusUnprocessableEntity).
						Build()

	ErrBadPagingToken = errors.NewBuilder("bad paging token").
				WithErrorCode(ErrCodeBadPagingToken).
				WithHTTPStatus(http.StatusBadRequest).
				Build()

	ErrAssetMismatch = errors.NewBuilder("Replacement record has a different ID or asset hash from original").
				WithErrorCode(ErrCodeAssetMismatch).
				WithHTTPStatus(http.StatusBadRequest).
				Build()

	ErrInvalidAssetHash = errors.NewBuilder("Invalid extension asset hash").
				WithErrorCode(ErrCodeInvalidAssetHash).
				WithHTTPStatus(http.StatusBadRequest).
				Build()

	ErrUnauthorized = errors.ErrUnauthorized

	ErrInvalidSubscriptionsSupportTransition = errors.NewBuilder("invalid subscription support level transition").
							WithErrorCode(ErrCodeInvalidSubscriptionsSupportTransition).
							WithHTTPStatus(http.StatusBadRequest).
							Build()

	ErrMonetizationRequired = errors.NewBuilder("channel is not monetized").
				WithErrorCode(ErrCodeChannelNotMonetized).
				WithHTTPStatus(http.StatusForbidden).
				Build()

	// ErrNullUploader is returned by a store when a load request can't be fulfilled
	ErrNullUploader = errors.NewBuilder("Null Uploader").
			WithErrorCode(ErrCodeNullUploader).
			WithHTTPStatus(http.StatusServiceUnavailable).
			Build()

	// ErrInvalidImageAssetsType is returned when an invalid image asset key for ImageAssetMap is provided
	ErrInvalidImageAssetsType = errors.NewBuilder("Invalid image asset type").
					WithErrorCode(ErrCodeInvalidImageAssetsType).
					WithHTTPStatus(http.StatusUnprocessableEntity).
					Build()

	// ErrInvalidUploadImageBody is returned when a request body doesn't match expectations
	ErrInvalidUploadImageBody = errors.NewBuilder("Invalid body. Must have 'logo', 'taskbar', 'discovery' and 'screenshot'").
					WithErrorCode(ErrCodeInvalidUploadImageBody).
					WithHTTPStatus(http.StatusBadRequest).
					Build()

	// ErrNotEnoughUploadLogo is returned when there is not enough logo to upload
	ErrNotEnoughUploadLogo = errors.NewBuilder("Not enough logo for upload, must >= 0").
				WithErrorCode(ErrCodeNotEnoughUploadLogo).
				WithHTTPStatus(http.StatusBadRequest).
				Build()

	// ErrNotEnoughUploadTaskbar is returned when there is not enough taskbar icon to upload
	ErrNotEnoughUploadTaskbar = errors.NewBuilder("Not enough taskbar icon for upload, must >= 0").
					WithErrorCode(ErrCodeNotEnoughUploadTaskbar).
					WithHTTPStatus(http.StatusBadRequest).
					Build()

	// ErrNotEnoughUploadDiscovery is returned when there is not enough discovery image to upload
	ErrNotEnoughUploadDiscovery = errors.NewBuilder("Not enough discovery, must >= 0").
					WithErrorCode(ErrCodeNotEnoughUploadDiscovery).
					WithHTTPStatus(http.StatusBadRequest).
					Build()

	// ErrNotEnoughUploadScreenshot is returned when there is not enough screenshot to upload
	ErrNotEnoughUploadScreenshot = errors.NewBuilder("Not enough screenshot, must >= 0").
					WithErrorCode(ErrCodeNotEnoughUploadScreenshot).
					WithHTTPStatus(http.StatusBadRequest).
					Build()

	// ErrTooManyUploadLogo is returned when there is too many logo to upload
	ErrTooManyUploadLogo = errors.NewBuilder("Too many logo, must <= 1").
				WithErrorCode(ErrCodeTooManyUploadLogo).
				WithHTTPStatus(http.StatusBadRequest).
				Build()

	// ErrTooManyUploadTaskbar is returned when there is too many taskbar icon to upload
	ErrTooManyUploadTaskbar = errors.NewBuilder("Too many screenshot, must <= 1").
				WithErrorCode(ErrCodeTooManyUploadTaskbar).
				WithHTTPStatus(http.StatusBadRequest).
				Build()

	// ErrTooManyUploadDiscovery is returned when there are too many discovery image to upload
	ErrTooManyUploadDiscovery = errors.NewBuilder("Too many discovery, must <= 1").
					WithErrorCode(ErrCodeTooManyUploadDiscovery).
					WithHTTPStatus(http.StatusBadRequest).
					Build()

	// ErrTooManyUploadScreenshot is returned when there are too many screenshot to upload
	ErrTooManyUploadScreenshot = errors.NewBuilder("Too many screenshot, existing + uploading screenshots must <= 6").
					WithErrorCode(ErrCodeTooManyUploadScreenshot).
					WithHTTPStatus(http.StatusBadRequest).
					Build()

	// ErrAssetNotFoundManifest is returned when the asset cannot be found in the Manifest
	ErrAssetNotFoundManifest = errors.NewBuilder("The asset was not found in manifest").
					WithErrorCode(ErrCodeAssetNotFoundManifest).
					WithHTTPStatus(http.StatusNotFound).
					Build()

	ErrInvalidImageUrl = errors.NewBuilder("invalid image url").
				WithErrorCode(ErrCodeInvalidImageUrl).
				WithHTTPStatus(http.StatusBadRequest).
				Build()

	ErrInvalidIconSize = errors.NewBuilder("invalid icon size").
				WithErrorCode(ErrCodeInvalidIconSize).
				WithHTTPStatus(http.StatusBadRequest).
				Build()

	ErrUsingRelativeWithAbsolutePaths = errors.NewBuilder("cannot mix relative and absolute paths").
						WithErrorCode(ErrCodeUsingRelativeWithAbsolutePaths).
						WithHTTPStatus(http.StatusConflict).
						Build()

	// ErrBitsIneligibleTransition is returned when a new extension manifest's desired Bits Feature Level isn't a valid
	// destination from the current Bits Feature Level
	ErrBitsIneligibleTransition = errors.NewBuilder("Invalid state transition").
					WithErrorCode(ErrCodeBitsIneligibleTransition).
					WithHTTPStatus(http.StatusBadRequest).
					Build()

	ErrVersionAlreadyExists = errors.NewBuilder("cannot clone to a version that already exists").
				WithErrorCode(ErrCodeVersionAlreadyExists).
				WithHTTPStatus(http.StatusConflict).
				Build()
)
