package middleware

import (
	"fmt"
	"net/http"
	"strings"
	"time"

	"code.justin.tv/chat/golibs/logx"

	"github.com/cactus/go-statsd-client/statsd"
	"goji.io/middleware"
	"goji.io/pat"
	"golang.org/x/net/context"
)

// Stats creates statsd timers and counters for request metrics.
func Stats(stats statsd.Statter, sampleRate float32) func(http.Handler) http.Handler {
	return func(h http.Handler) http.Handler {
		return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			ctx := r.Context()
			startTime := time.Now()
			statusWriter := &statusResponseWriter{w, 200}

			h.ServeHTTP(statusWriter, r)

			duration := time.Since(startTime)
			responseCode := statusWriter.status
			metricBases := metricNames(ctx, r)

			for _, metricBase := range metricBases {
				if err := stats.TimingDuration(fmt.Sprintf("%s.%d", metricBase, responseCode), duration, sampleRate); err != nil {
					logx.Warn(ctx, fmt.Sprintf("failed to report stats timing for %q", metricBase))
				}
				if err := stats.Inc(fmt.Sprintf("%s.%d", metricBase, responseCode), 1, sampleRate); err != nil {
					logx.Warn(ctx, fmt.Sprintf("failed to report stats counter for %q", metricBase))
				}
			}
		})
	}
}

func metricNames(ctx context.Context, r *http.Request) []string {
	match := middleware.Pattern(ctx)
	var pattern string
	if match == nil {
		pattern = "unknown-route"
	} else {
		// Replace pattern characters like / and : with graphite-friendly alternatives:
		// - /feeds/recent -> feeds-recent
		// - /feeds/:feed_id/posts -> feeds-_feed_id-posts
		pattern = match.(*pat.Pattern).String()
	}

	pattern = strings.TrimLeft(pattern, "/")
	pattern = strings.Replace(pattern, "/", "-", -1)
	pattern = strings.Replace(pattern, ":", "_", -1)

	source := GetSource(ctx)

	// Ensure the source name groups into one subfolder
	source = strings.Replace(source, ".", "_", -1)
	return []string{
		strings.Join([]string{"http", pattern, r.Method}, "."),
		strings.Join([]string{"httpsource", source, pattern, r.Method}, "."),
	}

}

type statusResponseWriter struct {
	http.ResponseWriter
	status int
}

func (sw *statusResponseWriter) WriteHeader(c int) {
	sw.ResponseWriter.WriteHeader(c)
	sw.status = c
}
