import datetime

from airflow import DAG
from airflow.operators.dummy_operator import DummyOperator
from piper.alerting.rollbar import rollbar_report_exc_info
from piper.operators.generic_db_transfer import GenericDbTransfer
from piper.operators.run_db_query import RunDbQuery
from piper.operators.clear_cache import ClearCache

from piper.queries.drops_campaigns_drops_overview import (INCREMENTAL_DELETE, INCREMENTAL,
                                                          UPDATE_DOMAIN_REPORT_META_DATA)
from piper.operators.validators.validate_drops_campaigns_drops_overview import ValidateDropsCampaignsDropsOverviewFields

from piper.utils import get_alert_email


default_args = {
    'owner': 'INS',
    'retries': 3,
    'retry_delay': datetime.timedelta(minutes=5),
    'start_date': datetime.datetime(2020, 8, 28),  # start date of first campaign
    'depends_on_past': False,
    'sla': datetime.timedelta(hours=2),
    'email': [get_alert_email()],
    'email_on_retry': False,
    'email_on_failure': False,
    'on_failure_callback': rollbar_report_exc_info
}

query_params = {
    'start_date': '{{ yesterday_ds }}',
    'end_date': '{{ ds }}',
}

with DAG('drops_campaigns_drops_overview_v2',
         default_args=default_args,
         schedule_interval='@daily',
         max_active_runs=1) as dag:

    # generate daily aggregations
    calculate_drops_overview_incremental = GenericDbTransfer(
        task_id='calculate_drops_overview_incremental',
        source_conn_id='piper-games',
        destination_conn_id='games-daily-incremental-writer',
        destination_table='drops_campaigns_drops_overview',
        preoperator=INCREMENTAL_DELETE,
        preoperator_params=query_params,
        sql=INCREMENTAL,
        query_params=query_params
    )

    # data validation
    validate_fields = ValidateDropsCampaignsDropsOverviewFields(task_id='validate_fields')

    # update report metadata table for piper-service to see min/max dates for this report
    update_domain_report_meta_data = RunDbQuery(
        task_id="update_domain_report_meta_data",
        conn_id="games-daily-incremental-writer",
        sql=UPDATE_DOMAIN_REPORT_META_DATA,
        query_params=query_params
    )

    # clears Redis cache in Piper Service so that it gets newest data for every request
    clear_cache = ClearCache(task_id='clear_cache', conn_id='redis')

    # done
    done = DummyOperator(task_id='done')

    # dependency order
    # pylint: disable=pointless-statement
    (calculate_drops_overview_incremental
     >> validate_fields >> update_domain_report_meta_data >> clear_cache >> done)
