import datetime

from airflow import DAG
from airflow.operators.dummy_operator import DummyOperator
from airflow.operators.sensors import ExternalTaskSensor
from piper.alerting.rollbar import rollbar_report_exc_info
from piper.queries.extensions_overview_v2 import (INCREMENTAL_DELETE, INCREMENTAL_SELECT, INCREMENTAL_FIELDS,
                                                  UPDATE_DOMAIN_REPORT_META_DATA)
from piper.operators.generic_db_transfer import GenericDbTransfer
from piper.operators.validators.validate_extensions_overview_v2 import (ValidateExtensionsOverviewV2Fields,
                                                                        ValidateExtensionsOverviewV2MovingAvg)
from piper.operators.run_db_query import RunDbQuery
from piper.operators.clear_cache import ClearCache
from piper.dates import get_execution_date
from piper.utils import get_alert_email

default_args = {
    'owner': 'INS',
    'retries': 3,
    'retry_delay': datetime.timedelta(minutes=5),
    'start_date': datetime.datetime(2018, 1, 31),
    'depends_on_past': False,
    'sla': datetime.timedelta(hours=2),  # email if SLA is missed
    'email': [get_alert_email()],
    'email_on_retry': False,
    'email_on_failure': False,
    'on_failure_callback': rollbar_report_exc_info,
}

with DAG('extensions_overview_v2',
         default_args=default_args,
         schedule_interval='0 1 * * *',
         max_active_runs=1) as dag:

    # Make sure that the previous DAG run is done before starting
    wait_yesterday_done = ExternalTaskSensor(
        task_id='wait_yesterday_done',
        external_dag_id='extensions_overview_v2',
        external_task_id='all_done',
        execution_date_fn=get_execution_date)

    # Aggregate report from Tahoe (Redshift) and place result in Aurora
    query_params = {'record_date': '{{ yesterday_ds }}'}
    calculate_overview_v2_incremental = GenericDbTransfer(
        task_id='calculate_overview_v2_incremental',
        source_conn_id='piper-games',  # from Redshift
        destination_conn_id='games-daily-incremental-writer',  # to Aurora
        destination_table='extensions_overview_v2',
        preoperator=INCREMENTAL_DELETE,  # cleanup previous records on Aurora (allow re-runs)
        preoperator_params=query_params,  # cleanup only within start/end dates
        sql=INCREMENTAL_SELECT,  # query to aggregate daily data from Redshift
        query_params=query_params,
        target_fields=INCREMENTAL_FIELDS)

    # Data validation
    validate_fields = ValidateExtensionsOverviewV2Fields(task_id='validate_fields')
    validate_moving_avg = ValidateExtensionsOverviewV2MovingAvg(task_id='validate_moving_avg')

    # update report metadata table for piper-service to see min/max dates for this report
    update_domain_report_meta_data = RunDbQuery(
        task_id='update_domain_report_meta_data',
        conn_id='games-daily-incremental-writer',
        sql=UPDATE_DOMAIN_REPORT_META_DATA,
        query_params={
            'start_date': '2018-01-31',
            'end_date': '{{ ds }}',
        })

    # clears Redis cache in Piper Service so that it gets newest data for every request
    clear_cache = ClearCache(task_id='clear_cache', conn_id='redis')
    # Done
    all_done = DummyOperator(task_id='all_done')

    # Setup task dependency order
    # pylint: disable=pointless-statement
    (wait_yesterday_done >> calculate_overview_v2_incremental >>
     validate_fields >> validate_moving_avg >>
     update_domain_report_meta_data >> clear_cache >> all_done)
