"""
Utility for sending rollbar alerts
"""
import sys

import rollbar
from airflow.settings import conf

from piper.utils import piper_environment

rollbar.init(conf.get('piper', 'rollbar_token'), piper_environment())


class LogLevel:
    ERROR = 'error'
    WARNING = 'warning'


def rollbar_report_exc_info(context):
    """
    Report a caught exception (from a try-catch block) to Rollbar with a DAG context.
    Note: The 'error' log level is configured in Rollbar to create a page in PagerDuty.
    """
    exc_info = sys.exc_info()  # current exception, is a tuple (exc_class, exc, trace)

    # Use exception class name on the fingerprint, which separates different exception, but keeps together
    # the same exception with different messages (e.g. if they include the game name in the message).
    exc_classname = str(exc_info[0])

    rollbar.report_exc_info(
        exc_info,
        level=LogLevel.ERROR,
        extra_data=build_extra_data(context),
        payload_data=build_payload_fingerprint(context, exc_classname))


def rollbar_report_message(context, message, level):
    """
    Send a log message to Rollbar with a DAG context for the payload and extra data.
    Note: The 'error' log level is configured in Rollbar to create a page in PagerDuty.
    """
    rollbar.report_message(
        message=message,
        level=level,
        extra_data=build_extra_data(context),
        payload_data=build_payload_fingerprint(context, message))


def build_extra_data(context):
    if context and 'task_instance' in context:
        task_instance = context['task_instance']
        execution_date = context['execution_date']
        return {
            'dag_id': task_instance.dag_id,
            'task_id': task_instance.task_id,
            'task_execution_date': execution_date.isoformat(),
            'task_duration': task_instance.duration,
            'task_state': task_instance.state
        }
    return None


def build_payload_fingerprint(context, message):
    fingerprint = build_fingerprint(context, message)
    if fingerprint:
        return {'fingerprint': fingerprint}
    return None


def build_fingerprint(context, message):
    """
    By default, rollbar builds a fingerprint automatically server-side
    which takes into account the exception type and some frame information from the traceback.

    We've chosen to use a custom fingerprinting for dag failures, where all report writes for a particular day of data
    are bucketed together, and all other dagruns are in their own bucket.  This will ensure that we are aggressively
    alerted of dag failures by rollbar, without the report writers spamming us with failures.
    """
    fingerprint = ''

    if context and 'task_instance' in context:
        task_instance = context['task_instance']
        execution_date = context['execution_date']

        dag_id = task_instance.dag_id  # e.g. "games_overview_v2"
        fingerprint += dag_id+'_'+execution_date.strftime("%Y-%m-%d")  # e.g. "games_overview_v2_2002-12-04"

    if message:
        fingerprint += '_'+message  # different messages count as different errors

    return fingerprint
