from datetime import datetime, timedelta

import pendulum
import pytest
from piper.dates import get_execution_date, is_backfill

utc = pendulum.timezone('UTC')
pst = pendulum.timezone('US/Pacific')


@pytest.mark.parametrize("now,expected", [
    (
        datetime(2019, 1, 11).replace(tzinfo=utc),
        datetime(2019, 1, 10).replace(tzinfo=utc)
    ),  # both utc
    (
        datetime(2019, 1, 11).replace(tzinfo=pst),
        datetime(2019, 1, 10).replace(tzinfo=utc)
    ),  # now has pst
    (
        datetime(2019, 1, 11),
        datetime(2019, 1, 10).replace(tzinfo=utc)
    ),  # now has no tzinfo
    (
        datetime(2019, 1, 11).replace(tzinfo=pst) + timedelta(hours=5, minutes=1),
        datetime(2019, 1, 10).replace(tzinfo=utc) + timedelta(hours=5, minutes=1)
    ),  # now has hours and minutes
])
def test_get_execution_date(now, expected):
    actual = get_execution_date(now)
    assert actual == expected


@pytest.mark.parametrize("today,report_execution_date", [
    (
        datetime(2019, 1, 18, 12, 30, 15, 419133).replace(tzinfo=utc),
        datetime(2019, 1, 17).replace(tzinfo=utc)
    ),
    (
        datetime(2019, 1, 18, 12, 30, 15, 419133).replace(tzinfo=utc),
        datetime(2019, 1, 17).replace(tzinfo=utc) + timedelta(hours=3)
    ),  # w hours
])
def test_is_not_backfill(today, report_execution_date):
    assert not is_backfill(report_execution_date, today)


@pytest.mark.parametrize("today,report_execution_date", [
    (
        datetime(2019, 1, 20, 12, 30, 15, 419133).replace(tzinfo=utc),
        datetime(2019, 1, 17).replace(tzinfo=utc)
    ),
    (
        datetime(2019, 1, 20, 12, 30, 15, 419133).replace(tzinfo=utc),
        datetime(2019, 1, 17).replace(tzinfo=utc) + timedelta(hours=3)
    ),  # w hours
])
def test_is_backfill(today, report_execution_date):
    assert is_backfill(report_execution_date, today)
