from piper.extensions_whitelist import EXTENSIONS_WHITELIST
from piper.operators.validators.base_postgres_data_validator import \
    BasePostgresDataValidator


# Validate fields on a whitelist of extensions
class ValidateExtensionsOverviewFields(BasePostgresDataValidator):
    block_dag_if_invalid = True
    conn_id = 'games-daily-incremental-reader'
    query = """
        SELECT * FROM extensions_overview
        WHERE record_date >= %(start_date)s AND record_date < %(end_date)s AND extension_id = ANY(%(whitelist)s)
    """
    query_params = {
        'start_date': '{{ yesterday_ds }}',
        'end_date': '{{ ds }}',
        'whitelist': EXTENSIONS_WHITELIST,
    }

    def validate(self):
        if not self.rows:
            self.add_error("The query returned no results, but most extensions from the whitelist should be in the DB")
            return

        self.row_id_field = 'extension_name'  # to show what extension failed the validation if there are errors

        self.validate_non_negative('installs')
        self.validate_non_negative('uninstalls')
        self.validate_non_negative('activates')
        self.validate_non_negative('total_views')
        self.validate_non_negative('unique_viewers')
        self.validate_non_negative('unique_interactors')
        self.validate_non_negative('total_clicks')
        self.validate_non_negative('clicks_per_interactor')
        self.validate_non_negative('interaction_rate')
        self.validate_non_negative('total_renders')
        self.validate_non_negative('unique_renderers')
        self.validate_non_negative('unique_active_channels')

        self.validate_greather_or_equal('total_renders', 'unique_renderers')


# Validate Moving average
class ValidateExtensionsOverviewMovingAvg(BasePostgresDataValidator):
    block_dag_if_invalid = False
    conn_id = 'games-daily-incremental-reader'
    query = """
        SELECT record_date,
          SUM(total_views) AS total_views,
          SUM(unique_viewers) AS unique_viewers,
          SUM(unique_interactors) AS unique_interactors,
          SUM(total_clicks) AS total_clicks,
          SUM(total_renders) AS total_renders,
          SUM(unique_renderers) AS unique_renderers
        FROM extensions_overview
        WHERE record_date >= %(start_date)s AND record_date < %(end_date)s
        GROUP BY record_date
        ORDER BY record_date DESC
    """
    query_params = {
        'start_date': '{{ macros.ds_add(ds, -90) }}',  # 90 days ago
        'end_date': '{{ ds }}',
    }

    def validate(self):
        if len(self.rows) < 90:
            self.log_info("Skip validation: loaded only {} rows, not enough data for moving average.", len(self.rows))
            return

        self.validate_rows_ordered_by_continuous_days('record_date')

        self.validate_moving_avg_bottom('total_views')
        self.validate_moving_avg_bottom('unique_viewers')
        self.validate_moving_avg_bottom('unique_interactors')
        self.validate_moving_avg_bottom('total_clicks')
        self.validate_moving_avg_bottom('total_renders')
        self.validate_moving_avg_bottom('unique_renderers')
