from piper.extensions_whitelist import EXTENSIONS_WHITELIST
from piper.operators.validators.base_postgres_data_validator import \
    BasePostgresDataValidator


# Validate fields on a whitelist of extensions
class ValidateExtensionsOverviewV2Fields(BasePostgresDataValidator):
    block_dag_if_invalid = True
    conn_id = 'games-daily-incremental-reader'
    query = """
        SELECT * FROM extensions_overview_v2
        WHERE record_date = %(record_date)s
        AND extension_id = ANY(%(whitelist)s)
    """
    query_params = {
        'record_date': '{{ yesterday_ds }}',
        'whitelist': EXTENSIONS_WHITELIST,
    }

    def validate(self):
        if not self.rows:
            self.add_error("The query returned no results, but most extensions from the whitelist should be in the DB")
            return

        self.row_id_field = 'extension_name'  # to show what extension failed the validation if there are errors

        self.validate_non_negative_fields()
        self.validate_greather_or_equal_fields()

    def validate_non_negative_fields(self):
        self.validate_non_negative('installs')
        self.validate_non_negative('uninstalls')
        self.validate_non_negative('activates')
        self.validate_non_negative('renders')
        self.validate_non_negative('renders_uniq_devices')
        self.validate_non_negative('renders_uniq_devices_7d')
        self.validate_non_negative('renders_uniq_devices_30d')
        self.validate_non_negative('renders_uniq_channels')
        self.validate_non_negative('renders_uniq_channels_7d')
        self.validate_non_negative('renders_uniq_channels_30d')
        self.validate_non_negative('views')
        self.validate_non_negative('views_uniq_devices')
        self.validate_non_negative('views_uniq_devices_7d')
        self.validate_non_negative('views_uniq_devices_30d')
        self.validate_non_negative('mouseenters')
        self.validate_non_negative('mouseenters_uniq_devices')
        self.validate_non_negative('mouseenters_uniq_devices_7d')
        self.validate_non_negative('mouseenters_uniq_devices_30d')
        self.validate_non_negative('clicks')
        self.validate_non_negative('clicks_uniq_devices')
        self.validate_non_negative('clicks_uniq_devices_7d')
        self.validate_non_negative('clicks_uniq_devices_30d')
        self.validate_non_negative('minimizns')
        self.validate_non_negative('minimizns_uniq_devices')
        self.validate_non_negative('unminimizns')
        self.validate_non_negative('unminimizns_uniq_devices')
        self.validate_non_negative('bits_spent')
        self.validate_non_negative('bits_transactions')
        self.validate_non_negative('bits_uniq_users')
        self.validate_non_negative('bits_uniq_users_7d')
        self.validate_non_negative('bits_uniq_users_30d')
        self.validate_non_negative('identity_links_uniq_users')
        self.validate_non_negative('identity_unlinks_uniq_users')
        self.validate_non_negative('ext_page_views')
        self.validate_non_negative('ext_page_views_uniq_devices')

    def validate_greather_or_equal_fields(self):
        self.validate_greather_or_equal('renders', 'renders_uniq_devices')
        self.validate_greather_or_equal('renders', 'renders_uniq_channels')
        self.validate_greather_or_equal('renders_uniq_devices_7d', 'renders_uniq_devices')
        self.validate_greather_or_equal('renders_uniq_devices_30d', 'renders_uniq_devices_7d')
        self.validate_greather_or_equal('renders_uniq_channels_7d', 'renders_uniq_channels')
        self.validate_greather_or_equal('renders_uniq_channels_30d', 'renders_uniq_channels_7d')
        self.validate_greather_or_equal('views', 'views_uniq_devices')
        self.validate_greather_or_equal('views_uniq_devices_7d', 'views_uniq_devices')
        self.validate_greather_or_equal('views_uniq_devices_30d', 'views_uniq_devices_7d')
        self.validate_greather_or_equal('mouseenters', 'mouseenters_uniq_devices')
        self.validate_greather_or_equal('mouseenters_uniq_devices_7d', 'mouseenters_uniq_devices')
        self.validate_greather_or_equal('mouseenters_uniq_devices_30d', 'mouseenters_uniq_devices_7d')
        self.validate_greather_or_equal('clicks', 'clicks_uniq_devices')
        self.validate_greather_or_equal('clicks_uniq_devices_7d', 'clicks_uniq_devices')
        self.validate_greather_or_equal('clicks_uniq_devices_30d', 'clicks_uniq_devices_7d')
        self.validate_greather_or_equal('minimizns', 'minimizns_uniq_devices')
        self.validate_greather_or_equal('unminimizns', 'unminimizns_uniq_devices')
        self.validate_greather_or_equal('bits_uniq_users_7d', 'bits_uniq_users')
        self.validate_greather_or_equal('bits_uniq_users_30d', 'bits_uniq_users_7d')
        self.validate_greather_or_equal('ext_page_views', 'ext_page_views_uniq_devices')


# Validate Moving average
class ValidateExtensionsOverviewV2MovingAvg(BasePostgresDataValidator):
    block_dag_if_invalid = False
    conn_id = 'games-daily-incremental-reader'
    query = """
        SELECT record_date,
          SUM(installs) AS installs,
          SUM(uninstalls) AS uninstalls,
          SUM(activates) AS activates,
          SUM(renders) AS renders,
          SUM(renders_uniq_devices) AS renders_uniq_devices,
          SUM(renders_uniq_devices_7d) AS renders_uniq_devices_7d,
          SUM(renders_uniq_devices_30d) AS renders_uniq_devices_30d,
          SUM(renders_uniq_channels) AS renders_uniq_channels,
          SUM(renders_uniq_channels_7d) AS renders_uniq_channels_7d,
          SUM(renders_uniq_channels_30d) AS renders_uniq_channels_30d,
          SUM(views) AS views,
          SUM(views_uniq_devices) AS views_uniq_devices,
          SUM(views_uniq_devices_7d) AS views_uniq_devices_7d,
          SUM(views_uniq_devices_30d) AS views_uniq_devices_30d,
          SUM(mouseenters) AS mouseenters,
          SUM(mouseenters_uniq_devices) AS mouseenters_uniq_devices,
          SUM(mouseenters_uniq_devices_7d) AS mouseenters_uniq_devices_7d,
          SUM(mouseenters_uniq_devices_30d) AS mouseenters_uniq_devices_30d,
          SUM(clicks) AS clicks,
          SUM(clicks_uniq_devices) AS clicks_uniq_devices,
          SUM(clicks_uniq_devices_7d) AS clicks_uniq_devices_7d,
          SUM(clicks_uniq_devices_30d) AS clicks_uniq_devices_30d,
          SUM(minimizns) AS minimizns,
          SUM(minimizns_uniq_devices) AS minimizns_uniq_devices,
          SUM(unminimizns) AS unminimizns,
          SUM(unminimizns_uniq_devices) AS unminimizns_uniq_devices,
          SUM(bits_spent) AS bits_spent,
          SUM(bits_transactions) AS bits_transactions,
          SUM(bits_uniq_users) AS bits_uniq_users,
          SUM(bits_uniq_users_7d) AS bits_uniq_users_7d,
          SUM(bits_uniq_users_30d) AS bits_uniq_users_30d,
          SUM(identity_links_uniq_users) AS identity_links_uniq_users,
          SUM(identity_unlinks_uniq_users) AS identity_unlinks_uniq_users,
          SUM(ext_page_views) AS ext_page_views,
          SUM(ext_page_views_uniq_devices) AS ext_page_views_uniq_devices
        FROM extensions_overview_v2
        WHERE record_date >= %(start_date)s AND record_date < %(end_date)s
        GROUP BY record_date
        ORDER BY record_date DESC
    """
    query_params = {
        'start_date': '{{ macros.ds_add(ds, -90) }}',  # 90 days ago
        'end_date': '{{ ds }}',
    }

    def validate(self):
        if len(self.rows) < 90:
            self.log_info("Skip validation: loaded only {} rows, not enough data for moving average.", len(self.rows))
            return

        self.validate_rows_ordered_by_continuous_days('record_date')

        self.validate_moving_avg_bottom('installs')
        self.validate_moving_avg_bottom('uninstalls')
        self.validate_moving_avg_bottom('activates')
        self.validate_moving_avg_bottom('renders')
        self.validate_moving_avg_bottom('renders_uniq_devices')
        self.validate_moving_avg_bottom('renders_uniq_devices_7d')
        self.validate_moving_avg_bottom('renders_uniq_devices_30d')
        self.validate_moving_avg_bottom('renders_uniq_channels')
        self.validate_moving_avg_bottom('renders_uniq_channels_7d')
        self.validate_moving_avg_bottom('renders_uniq_channels_30d')
        self.validate_moving_avg_bottom('views')
        self.validate_moving_avg_bottom('views_uniq_devices')
        self.validate_moving_avg_bottom('views_uniq_devices_7d')
        self.validate_moving_avg_bottom('views_uniq_devices_30d')
        self.validate_moving_avg_bottom('mouseenters')
        self.validate_moving_avg_bottom('mouseenters_uniq_devices')
        self.validate_moving_avg_bottom('mouseenters_uniq_devices_7d')
        self.validate_moving_avg_bottom('mouseenters_uniq_devices_30d')
        self.validate_moving_avg_bottom('clicks')
        self.validate_moving_avg_bottom('clicks_uniq_devices')
        self.validate_moving_avg_bottom('clicks_uniq_devices_7d')
        self.validate_moving_avg_bottom('clicks_uniq_devices_30d')
        self.validate_moving_avg_bottom('minimizns')
        self.validate_moving_avg_bottom('minimizns_uniq_devices')
        self.validate_moving_avg_bottom('unminimizns')
        self.validate_moving_avg_bottom('unminimizns_uniq_devices')
        self.validate_moving_avg_bottom('bits_spent')
        self.validate_moving_avg_bottom('bits_transactions')
        self.validate_moving_avg_bottom('bits_uniq_users')
        self.validate_moving_avg_bottom('bits_uniq_users_7d')
        self.validate_moving_avg_bottom('bits_uniq_users_30d')
        self.validate_moving_avg_bottom('identity_links_uniq_users')
        self.validate_moving_avg_bottom('identity_unlinks_uniq_users')
        self.validate_moving_avg_bottom('ext_page_views')
        self.validate_moving_avg_bottom('ext_page_views_uniq_devices')
