from piper.games_whitelist import GAMES_WHITELIST
from piper.operators.validators.base_postgres_data_validator import \
    BasePostgresDataValidator


# Validate fields on a whitelist of games
class ValidateGamesOverviewFields(BasePostgresDataValidator):
    block_dag_if_invalid = True
    conn_id = 'games-daily-incremental-reader'
    query = """
        SELECT * FROM games_overview
        WHERE record_date >= %(start_date)s AND record_date < %(end_date)s AND game = ANY(%(whitelist)s)
    """
    query_params = {
        'start_date': '{{ yesterday_ds }}',
        'end_date': '{{ ds }}',
        'whitelist': GAMES_WHITELIST,
    }

    def validate(self):
        if not self.rows:
            self.add_error("The query returned no results, but most games from the whitelist should be in the DB")
            return

        self.row_id_field = 'game'  # to show what game failed the validation if there are errors

        self.validate_non_negative('non_live_views')
        self.validate_non_negative('live_uniques')
        self.validate_non_negative('non_live_uniques')
        self.validate_non_negative('live_hours')
        self.validate_non_negative('non_live_hours')
        self.validate_non_negative('unique_broadcasters')
        self.validate_non_negative('hours_broadcast')

        self.validate_greather_or_equal('live_views', 'live_uniques')
        self.validate_greather_or_equal('non_live_views', 'non_live_uniques')


# Validate Moving average
class ValidateGamesOverviewMovingAvg(BasePostgresDataValidator):
    block_dag_if_invalid = False
    conn_id = 'games-daily-incremental-reader'
    query = """
        SELECT record_date,
            SUM(live_views)   AS live_views,   SUM(non_live_views)   AS non_live_views,
            SUM(live_uniques) AS live_uniques, SUM(non_live_uniques) AS non_live_uniques,
            SUM(live_hours)   AS live_hours,   SUM(non_live_hours)   AS non_live_hours,
            SUM(unique_broadcasters) AS unique_broadcasters, SUM(hours_broadcast) AS hours_broadcast
        FROM games_overview
        WHERE record_date >= %(start_date)s AND record_date < %(end_date)s
        GROUP BY record_date
        ORDER BY record_date DESC
    """
    query_params = {
        'start_date': '{{ macros.ds_add(ds, -90) }}',  # 90 days ago
        'end_date': '{{ ds }}',
    }

    def validate(self):
        if len(self.rows) < 90:
            self.log_info("Skip validation: loaded only {} rows, not enough data for moving average.", len(self.rows))
            return

        self.validate_rows_ordered_by_continuous_days('record_date')

        self.validate_moving_avg_bottom('live_views')
        self.validate_moving_avg_bottom('non_live_views')
        self.validate_moving_avg_bottom('live_uniques')
        self.validate_moving_avg_bottom('non_live_uniques')
        self.validate_moving_avg_bottom('live_hours')
        self.validate_moving_avg_bottom('non_live_hours')
        self.validate_moving_avg_bottom('unique_broadcasters')
        self.validate_moving_avg_bottom('hours_broadcast')
