from piper.games_whitelist import GAMES_WHITELIST
from piper.operators.validators.base_postgres_data_validator import \
    BasePostgresDataValidator


# Validate fields on a whitelist of games
class ValidateGamesOverviewV2Fields(BasePostgresDataValidator):
    block_dag_if_invalid = True
    conn_id = 'games-daily-incremental-reader'
    query = """
        SELECT * FROM games_overview_v2
        WHERE record_date >= %(start_date)s AND record_date < %(end_date)s AND game = ANY(%(whitelist)s)
    """
    query_params = {
        'start_date': '{{ yesterday_ds }}',
        'end_date': '{{ ds }}',
        'whitelist': GAMES_WHITELIST,
    }

    def validate(self):
        self.row_id_field = 'game'  # to show what game failed the validation if there are errors

        self.validate_non_negative('live_views')
        self.validate_non_negative('non_live_views')
        self.validate_non_negative('total_unique_viewers')
        self.validate_non_negative('live_uniques')
        self.validate_non_negative('non_live_uniques')
        self.validate_non_negative('average_concurrent_viewers')
        self.validate_non_negative('peak_concurrent_viewers')
        self.validate_non_negative('live_hours')
        self.validate_non_negative('non_live_hours')
        self.validate_non_negative('unique_broadcasters')
        self.validate_non_negative('hours_broadcast')
        self.validate_non_negative('average_concurrent_broadcasters')
        self.validate_non_negative('peak_concurrent_broadcasters')
        self.validate_non_negative('live_unique_chat_participants')
        self.validate_non_negative('total_live_chat_messages_sent')
        self.validate_non_negative('unique_active_channels_with_extension')
        self.validate_non_negative('clips_created')
        self.validate_non_negative('clip_views')

        self.validate_greather_or_equal('live_views', 'live_uniques')
        self.validate_greather_or_equal('non_live_views', 'non_live_uniques')
        self.validate_greather_or_equal('peak_concurrent_viewers', 'average_concurrent_viewers')
        self.validate_greather_or_equal('peak_concurrent_broadcasters', 'average_concurrent_broadcasters')
        self.validate_greather_or_equal('total_live_chat_messages_sent', 'live_unique_chat_participants')

        # other fields (dates and urls)
        #   unique_active_extensions          (is Null for now)
        #   peak_time_concurrent_broadcasters (is a time in minutes within the last day)
        #   peak_time_concurrent_viewers'     (is a time in minutes within the last day)
        #   top_clip_url                      (is a URL)
        #   top_clip_url_embe                 (is a URL)


# Validate Moving average
class ValidateGamesOverviewV2MovingAvg(BasePostgresDataValidator):
    block_dag_if_invalid = False
    conn_id = 'games-daily-incremental-reader'
    query = """
        SELECT record_date,
            SUM(live_views) AS live_views,
            SUM(non_live_views) AS non_live_views,
            SUM(live_uniques) AS live_uniques,
            SUM(non_live_uniques) AS non_live_uniques,
            SUM(live_hours) AS live_hours,
            SUM(non_live_hours) AS non_live_hours,
            SUM(unique_broadcasters) AS unique_broadcasters,
            SUM(hours_broadcast) AS hours_broadcast,
            SUM(live_unique_chat_participants) AS live_unique_chat_participants,
            SUM(total_live_chat_messages_sent) AS total_live_chat_messages_sent,
            SUM(unique_active_channels_with_extension) AS unique_active_channels_with_extension,
            SUM(clips_created) AS clips_created,
            SUM(clip_views) AS clip_views
        FROM games_overview_v2
        WHERE record_date >= %(start_date)s AND record_date < %(end_date)s
        GROUP BY record_date
        ORDER BY record_date DESC
    """
    query_params = {
        'start_date': '{{ macros.ds_add(ds, -90) }}',  # 90 days ago
        'end_date': '{{ ds }}',
    }

    def validate(self):
        if len(self.rows) < 90:
            self.log_info("Skip validation: loaded only {} rows, not enough data for moving average.", len(self.rows))
            return

        self.validate_rows_ordered_by_continuous_days('record_date')

        self.validate_moving_avg_bottom('live_views')
        self.validate_moving_avg_bottom('non_live_views')
        self.validate_moving_avg_bottom('live_uniques')
        self.validate_moving_avg_bottom('non_live_uniques')
        self.validate_moving_avg_bottom('live_hours')
        self.validate_moving_avg_bottom('non_live_hours')
        self.validate_moving_avg_bottom('unique_broadcasters')
        self.validate_moving_avg_bottom('hours_broadcast')
        self.validate_moving_avg_bottom('live_unique_chat_participants')
        self.validate_moving_avg_bottom('total_live_chat_messages_sent')
        self.validate_moving_avg_bottom('unique_active_channels_with_extension')
        self.validate_moving_avg_bottom('clips_created')
        self.validate_moving_avg_bottom('clip_views')
