# Delete previous days for same time range before inserting data. Prevents duplicates from being inserted
INCREMENTAL_DELETE = """
    DELETE FROM drops_campaigns_drops_overview WHERE record_date >= %(start_date)s AND record_date < %(end_date)s
"""

# Query Redshift and generate daily metrics in the time range
INCREMENTAL = """
WITH
drops_community_callout_actions as (
SELECT time_utc, campaign_id, user_id, channel_id, location, action,
    CASE drop_type
        WHEN 'time_based' THEN 'mw_event'
        WHEN 'event_based' THEN 'event_based_event'
    END AS
    drop_type
FROM tahoe.drops_community_callout_actions
WHERE "date" >= DATEADD(day, -1, %(start_date)s) AND "date" < DATEADD(day, 1, %(start_date)s)
AND drop_type != ''
)

, drops_server_campaigns AS (
SELECT campaignid AS campaign_id, gameid AS game_id, name,
TIMESTAMP 'epoch' + startdateunix * INTERVAL '1 second' as start_date,
TIMESTAMP 'epoch' + enddateunix * INTERVAL '1 second' as end_date
FROM dbsnapshots.drops2_campaigns
)

, drops_server_drops AS (
SELECT campaignid AS campaign_id, dropid AS drop_id, name,
CASE droptype
    WHEN 'TIME_BASED' THEN 'mw_event'
    WHEN 'EVENT_BASED' THEN 'event_based_event'
END AS drop_type
FROM dbsnapshots.drops2_drops
)

, drops_campaigns_metadata as (
SELECT a.campaign_id, game_id, a.name as campaign_name, drop_id, b.name as drop_name, drop_type
FROM drops_server_campaigns a
LEFT JOIN drops_server_drops b
    ON a.campaign_id = b.campaign_id
)

, drops_mw_goal_complete AS (
SELECT a.*, b.drop_name
FROM tahoe.drops_mw_goal_complete a
LEFT JOIN drops_campaigns_metadata b
    ON a.campaign_id = b.campaign_id
    AND a.drop_id = b.drop_id
WHERE "date" >= DATEADD(day, -1, %(start_date)s) AND "date" < DATEADD(day, 1, %(start_date)s)
)

, drops_claimed as (
SELECT a.*, b.drop_name
FROM tahoe.drops_claimed a
LEFT JOIN drops_campaigns_metadata b
    ON a.campaign_id = b.campaign_id
    AND a.drop_id = b.drop_id
WHERE "date" >= DATEADD(day, -1, %(start_date)s) AND "date" < DATEADD(day, 1, %(start_date)s)
)

, drops_client_claimable_impression AS (
SELECT a.*, b.drop_type, b.drop_name, b.game_id
FROM tahoe.drops_client_claimable_impression a
LEFT JOIN drops_campaigns_metadata b
    ON a.campaign_id = b.campaign_id
    AND a.drop_id = b.drop_id
WHERE "date" >= DATEADD(day, -1, %(start_date)s) AND "date" < DATEADD(day, 1, %(start_date)s)
)

, callout_with_games AS (
SELECT a.time_utc, a.channel_id, a.user_id, a.campaign_id, b.name as campaign_name,
a.drop_type, a.location, a.action, b.game_id
FROM drops_community_callout_actions a
LEFT JOIN drops_server_campaigns b
    on a.campaign_id = b.campaign_id
WHERE game_id != 0
AND %(start_date)s >= DATE_TRUNC('day', b.start_date) and %(start_date)s < b.end_date
GROUP BY 1,2,3,4,5,6,7,8,9
)

, viewers_eligible AS (
SELECT campaign_id, campaign_name, drop_type, game_id,
count(distinct user_id) AS viewers_eligible
FROM callout_with_games
WHERE time_utc >= %(start_date)s  AND time_utc < DATEADD(day, 1, %(start_date)s)
    AND location = 'drops_info'
    AND action = 'render'
GROUP BY 1,2,3,4
)

, viewers_eligible_viewers AS (
SELECT campaign_id, drop_type, game_id, user_id
FROM callout_with_games
WHERE time_utc >= %(start_date)s  AND time_utc < DATEADD(day, 1, %(start_date)s)
    AND location = 'drops_info'
    AND action = 'render'
GROUP BY 1,2,3,4
)

, streamers_eligible AS (
SELECT campaign_id, drop_type, game_id,
count(distinct channel_id) AS streamers_eligible
FROM callout_with_games
WHERE time_utc >= %(start_date)s  AND time_utc < DATEADD(day, 1, %(start_date)s)
    AND location = 'drops_info'
    AND action = 'render'
GROUP BY 1,2,3
)

, event_drops_earned AS (
SELECT a.campaign_id, a.drop_id, a.drop_name, a.drop_type, a.game_id,
count(distinct a.user_id) AS event_drops_earned
FROM drops_client_claimable_impression a
JOIN viewers_eligible_viewers b
    ON a.campaign_id = b.campaign_id
    AND a.game_id = b.game_id
    AND a.drop_type = b.drop_type
    AND a.user_id = b.user_id
WHERE a.time_utc >= %(start_date)s AND a.time_utc < DATEADD(day, 1, %(start_date)s)
AND a.drop_type = 'event_based_event'
GROUP BY 1,2,3,4,5
)

, mw_drops_earned AS (
SELECT a.campaign_id, a.drop_id, a.drop_name, a.drop_type, a.game_id,
count(distinct a.user_id) AS mw_drops_earned
FROM drops_mw_goal_complete a
JOIN viewers_eligible_viewers b
    ON a.campaign_id = b.campaign_id
    AND a.game_id = b.game_id
    AND a.drop_type = b.drop_type
    AND a.user_id = b.user_id
WHERE a.time_utc >= %(start_date)s  AND a.time_utc < DATEADD(day, 1, %(start_date)s)
GROUP BY 1,2,3,4,5
)

, viewers_claimed as (
SELECT campaign_id, drop_id, drop_name, drop_type, game_id,
count(distinct user_id) AS viewers_claimed
FROM drops_claimed
WHERE time_utc >= %(start_date)s AND time_utc < DATEADD(day, 1, %(start_date)s)
GROUP BY 1,2,3,4,5
)

, viewers_linked_and_claimed as (
SELECT campaign_id, drop_id, drop_name, drop_type, game_id,
count(distinct user_id) AS viewers_linked_and_claimed
FROM drops_claimed
WHERE time_utc >= %(start_date)s AND time_utc < DATEADD(day, 1, %(start_date)s)
    AND account_linked = true
GROUP BY 1,2,3,4,5
)

, mw_drops_rows as (
SELECT
a.campaign_id,
a.campaign_name,
a.game_id,
a.drop_type,
c.drop_id,
c.drop_name,
streamers_eligible,
viewers_eligible,
mw_drops_earned,
viewers_claimed,
viewers_linked_and_claimed
FROM viewers_eligible a
LEFT JOIN streamers_eligible b
    ON a.campaign_id = b.campaign_id
    AND a.game_id = b.game_id
    AND a.drop_type = b.drop_type
LEFT JOIN mw_drops_earned c
    ON a.campaign_id = c.campaign_id
    AND a.game_id = c.game_id
    AND a.drop_type = c.drop_type
LEFT JOIN viewers_claimed d
    ON a.campaign_id = d.campaign_id
    AND a.game_id = d.game_id
    AND a.drop_type = d.drop_type
    AND c.drop_id = d.drop_id
LEFT JOIN viewers_linked_and_claimed e
    ON a.campaign_id = e.campaign_id
    AND a.game_id = e.game_id
    AND a.drop_type = e.drop_type
    AND c.drop_id = e.drop_id
GROUP BY 1,2,3,4,5,6,7,8,9,10,11
)

, event_drops_rows as (
SELECT
a.campaign_id,
a.campaign_name,
a.game_id,
a.drop_type,
c.drop_id,
c.drop_name,
streamers_eligible,
viewers_eligible,
event_drops_earned,
viewers_claimed,
viewers_linked_and_claimed
FROM viewers_eligible a
LEFT JOIN streamers_eligible b
    ON a.campaign_id = b.campaign_id
    AND a.game_id = b.game_id
    AND a.drop_type = b.drop_type
LEFT JOIN event_drops_earned c
    ON a.campaign_id = c.campaign_id
    AND a.game_id = c.game_id
    AND a.drop_type = c.drop_type
LEFT JOIN viewers_claimed d
    ON a.campaign_id = d.campaign_id
    AND a.game_id = d.game_id
    AND a.drop_type = d.drop_type
    AND c.drop_id = d.drop_id
LEFT JOIN viewers_linked_and_claimed e
    ON a.campaign_id = e.campaign_id
    AND a.game_id = e.game_id
    AND a.drop_type = e.drop_type
    AND c.drop_id = e.drop_id
GROUP BY 1,2,3,4,5,6,7,8,9,10,11
)

, claimed_drops_rows as (
SELECT
a.campaign_id,
a.campaign_name,
a.game_id,
a.drop_type,
c.drop_id,
c.drop_name,
streamers_eligible,
viewers_eligible,
viewers_claimed,
viewers_linked_and_claimed
FROM viewers_eligible a
LEFT JOIN streamers_eligible b
    ON a.campaign_id = b.campaign_id
    AND a.game_id = b.game_id
    AND a.drop_type = b.drop_type
LEFT JOIN viewers_claimed c
    ON a.campaign_id = c.campaign_id
    AND a.game_id = c.game_id
    AND a.drop_type = c.drop_type
LEFT JOIN viewers_linked_and_claimed d
    ON a.campaign_id = d.campaign_id
    AND a.game_id = d.game_id
    AND a.drop_type = d.drop_type
    AND c.drop_id = d.drop_id
GROUP BY 1,2,3,4,5,6,7,8,9,10
)

SELECT
%(start_date)s::DATE AS record_date,
coalesce(agg.campaign_id, c.campaign_id) AS campaign_id,
coalesce(agg.drop_id, c.drop_id) AS drop_id,
coalesce(agg.game_id, c.game_id) AS game_id,
metadata.name as game_name,
coalesce(agg.streamers_eligible, c.streamers_eligible, 0) AS streamers_eligible,
coalesce(event_drops_earned, 0) AS event_drops_earned,
coalesce(mw_drops_earned, 0) AS mw_drops_earned,
coalesce(agg.viewers_claimed, c.viewers_claimed, 0) AS viewers_claimed,
coalesce(agg.viewers_linked_and_claimed, c.viewers_linked_and_claimed, 0) AS viewers_linked_and_claimed,
coalesce(agg.campaign_name, c.campaign_name) AS campaign_name,
coalesce(agg.drop_name, c.drop_name) AS drop_name
FROM (
    SELECT
    coalesce(a.campaign_id, b.campaign_id) AS campaign_id,
    coalesce(a.campaign_name, b.campaign_name) AS campaign_name,
    coalesce(a.drop_id, b.drop_id) AS drop_id,
    coalesce(a.drop_name, b.drop_name) AS drop_name,
    coalesce(a.game_id, b.game_id) AS game_id,
    coalesce(a.viewers_eligible, b.viewers_eligible, 0) AS viewers_eligible,
    coalesce(a.viewers_claimed, b.viewers_claimed, 0) AS viewers_claimed,
    coalesce(a.viewers_linked_and_claimed, b.viewers_linked_and_claimed, 0) AS viewers_linked_and_claimed,
    coalesce(a.streamers_eligible, b.streamers_eligible, 0) AS streamers_eligible,
    coalesce(event_drops_earned, 0) AS event_drops_earned,
    coalesce(mw_drops_earned, 0) AS mw_drops_earned
    FROM mw_drops_rows a
    FULL OUTER JOIN event_drops_rows b
        ON a.campaign_id = b.campaign_id
        AND a.game_id = b.game_id
        AND a.drop_id = b.drop_id
    GROUP BY 1,2,3,4,5,6,7,8,9,10,11
) agg
FULL OUTER JOIN claimed_drops_rows c
    ON agg.campaign_id = c.campaign_id
    AND agg.game_id = c.game_id
    AND agg.drop_id = c.drop_id
LEFT JOIN dbsnapshots.games metadata
    ON coalesce(agg.game_id, c.game_id) = metadata.id
WHERE coalesce(agg.drop_id, c.drop_id) != ''
"""


UPDATE_DOMAIN_REPORT_META_DATA = """
BEGIN;
DELETE FROM latest_domain_report_meta_data WHERE domain = 'drops' AND report = 'campaigns_drops_overview';
INSERT INTO latest_domain_report_meta_data
SELECT %(end_date)s
    , 'drops'
    , 'campaigns_drops_overview'
    , campaign_id
    , campaign_name
    , MIN(record_date)
    , MAX(record_date)
FROM drops_campaigns_drops_overview
WHERE record_date >= '2020-03-20' AND record_date < %(end_date)s
GROUP BY campaign_id, campaign_name;
COMMIT;
"""
