# Delete previous rows for the same record_date before calculating new aggregations,
# so the DAG run is idempotent and can be run multiple times with the same result.
INCREMENTAL_DELETE = """
DELETE FROM games_overview_v2 WHERE record_date = %(record_date)s
"""

# Common condition to partition Tahoe events by date and filter by time_utc with the given record_date.
# The record_date param is a date in UTC (e.g. "2020-10-22"); but Tahoe dates are in PT (Pacific Time), so the
# partition needs an extra day, we need to load 2 PT day partitions, to fit one UTC day on it.
time_utc_in_record_date = """
        "date" >= DATEADD(day, -1, %(record_date)s) AND "date"   < DATEADD(day, 1, %(record_date)s)
        AND time_utc >= %(record_date)s             AND time_utc < DATEADD(day, 1, %(record_date)s)
"""

# Query Redshift to extract daily metrics for games in the time interval.
INCREMENTAL = """
WITH

active_game_ids AS (
  SELECT game_id,
         avg_ccu_past_7_days AS ccu7
  FROM cubes."daily_ccu_summary_by_game" -- generated daily from tahoe data
  WHERE day = %(record_date)s
    AND avg_ccu_past_7_days > 0.01
),
active_games_ranked AS (
    SELECT g.id AS id, -- game/category id
           LOWER(g.name) AS name, -- official category name
           -- Resolve duplicates (pick highest CCUs) ref https://jira.twitch.com/browse/E2-494
           -- dbsnapshots.categories should resolve duplicates eventually, then cleanup this
           ROW_NUMBER() OVER (PARTITION BY LOWER(g.name) ORDER BY ag.ccu7 DESC) AS rank
    FROM dbsnapshots.categories g
    JOIN active_game_ids ag ON g.id = ag.game_id -- only active games
),
active_games AS (
  SELECT id, name FROM active_games_ranked WHERE rank = 1
),
category_aliases AS (
  SELECT category_id,
         CAST(LOWER(a.alias) AS VARCHAR(64)) AS alias
  FROM dbsnapshots.category_aliases a
),
game_aliases AS (
    SELECT g.id, g.name, a.alias
    FROM active_games g JOIN category_aliases a ON g.id = a.category_id
    UNION
    SELECT g.id, g.name, g.name AS alias
    FROM active_games g
),

-- For testing, you can use this small static list instead
-- WITH
-- active_games AS (
--     SELECT 138585 as id, 'hearthstone' as name  UNION
--     SELECT 488552 as id, 'overwatch' as name
-- ),
-- game_aliases AS (
--     SELECT 138585 as id, 'hearthstone' as name, 'hearthstone' as alias UNION
--     SELECT 138585 as id, 'hearthstone' as name, 'hs' as alias UNION
--     SELECT 138585 as id, 'hearthstone' as name, 'hearthstone: heroes of warcraft' as alias UNION
--     SELECT 138585 as id, 'hearthstone' as name, 'rng please' as alias UNION
--     SELECT 138585 as id, 'hearthstone' as name, '炉石传说' as alias UNION
--     SELECT 138585 as id, 'hearthstone' as name, '하스스톤' as alias UNION
--     SELECT 138585 as id, 'hearthstone' as name, 'ハースストーン' as alias UNION
--     SELECT 488552 as id, 'overwatch' as name, 'overwatch' as alias UNION
--     SELECT 488552 as id, 'overwatch' as name, 'overwatch: origins edition' as alias UNION
--     SELECT 488552 as id, 'overwatch' as name, 'play of the game' as alias UNION
--     SELECT 488552 as id, 'overwatch' as name, 'オーバーウォッチ' as alias UNION
--     SELECT 488552 as id, 'overwatch' as name, '鬥陣特攻' as alias UNION
--     SELECT 488552 as id, 'overwatch' as name, '오버워치' as alias UNION
--     SELECT 488552 as id, 'overwatch' as name, '守望先锋' as alias
-- ),

video_play AS (
    SELECT ga.id AS game_id,
        device_id,
        live,
        vod_type,
        content_mode,
        url
    FROM tahoe."video-play" vp
    INNER JOIN game_aliases ga ON CAST(LOWER(vp.game) AS VARCHAR(64)) = ga.alias
    WHERE {time_utc_in_record_date}
),

views_aggr AS (
    SELECT vp.game_id,
        SUM(CASE WHEN vp.live=TRUE OR vp.live IS NULL THEN 1 ELSE 0 END) AS live_views,
        SUM(CASE WHEN vp.live=FALSE AND NVL(vp.vod_type,'') != 'clip' THEN 1 ELSE 0 END) AS non_live_views,
        COUNT(DISTINCT (CASE WHEN vp.live=TRUE OR vp.live IS NULL THEN vp.device_id END)) AS live_unique,
        COUNT(DISTINCT (CASE WHEN vp.live=FALSE AND NVL(vp.vod_type,'') != 'clip' THEN vp.device_id END))
            AS non_live_unique,
        SUM(CASE WHEN NVL(vp.vod_type, '') = 'clip' THEN 1 ELSE 0 END) AS clips_views,
        COUNT(DISTINCT (CASE WHEN NVL(vp.vod_type,'') != 'clip' THEN vp.device_id END)) AS total_unique_viewers
    FROM video_play vp
    GROUP BY 1
),

minute_broadcast AS (
    SELECT ga.id AS game_id,
        channel_id,
        DATE_TRUNC('minute', time_utc) AS time_minute
    FROM tahoe."minute_broadcast" mb
    INNER JOIN game_aliases ga ON CAST(LOWER(mb.game) AS VARCHAR(64)) = ga.alias
    WHERE {time_utc_in_record_date}
),

broadcasters AS (
    SELECT mb.game_id,
      TRUNC(COUNT(*)::FLOAT/60, 2) AS hours_broadcasted,
      COUNT(DISTINCT channel_id) AS uniques
    FROM minute_broadcast mb
    GROUP BY 1
),

broadcasters_by_minute AS (
    SELECT game_id,
        time_minute,
        COUNT(DISTINCT channel_id) AS uniques,
        ROW_NUMBER() OVER(PARTITION BY game_id
            ORDER BY uniques DESC, time_minute ASC) AS uniques_rank -- rank 1 is the minute with max channels
    FROM minute_broadcast
    GROUP BY 1, 2
),

broadcasters_concurrents AS (
    SELECT game_id,
        AVG(uniques) AS avg_concurrents,
        MAX(uniques) AS max_concurrents
    FROM broadcasters_by_minute
    GROUP BY 1
),

broadcasters_concurrents_peak AS (
    SELECT game_id,
        time_minute AS peak_time_minute
    FROM broadcasters_by_minute
    WHERE uniques_rank = 1
),

minute_watched AS (
    SELECT ga.id AS game_id,
        channel_id,
        live,
        vod_type,
        DATE_TRUNC('minute', time_utc) AS time_minute
    FROM tahoe."minute-watched" mw
    INNER JOIN game_aliases ga ON CAST(LOWER(mw.game) AS VARCHAR(64)) = ga.alias
    WHERE {time_utc_in_record_date}
),

hours_watched_live_aggr AS (
    SELECT mw.game_id,
        COUNT(*) AS minutes_watched_live,
        TRUNC(minutes_watched_live::FLOAT/60, 2) AS hours_watched_live
    FROM minute_broadcast mb
    INNER JOIN minute_watched mw
        ON mw.channel_id = mb.channel_id AND mw.time_minute = mb.time_minute
    WHERE mw.live = TRUE OR mw.live IS NULL
    GROUP BY 1
),

hours_watched_nonlive_aggr AS (
    SELECT mw.game_id,
        COUNT(*) AS minutes_watched_non_live,
        TRUNC(minutes_watched_non_live::FLOAT/60, 2) AS hours_watched_non_live
    FROM minute_watched mw
    WHERE mw.live = FALSE AND NVL(mw.vod_type,'') != 'clip'
    GROUP BY 1
),

chat_messages AS (
    SELECT channel_id,
        user_id,
        DATE_TRUNC('minute', time_utc) AS time_minute
    FROM tahoe."server_chat_message"
    WHERE {time_utc_in_record_date}
),

chat_live_aggregate AS (
    SELECT mb.game_id,
        COUNT(*) AS total_live_chat_messages_sent,
        COUNT(DISTINCT ch.user_id) AS live_unique_chat_participants
    FROM minute_broadcast mb
    INNER JOIN chat_messages ch
        ON ch.channel_id = mb.channel_id AND ch.time_minute = mb.time_minute
    GROUP BY 1
),

liveline_channel AS (
    -- Ensure only one view count per game/channel/minute
    SELECT
        channel_category_id AS game_id,
        channel_id,
        DATE_TRUNC('minute', time_utc) AS time_minute,
        MAX(viewcount) AS ccu
    FROM tahoe."liveline_channel"
    WHERE {time_utc_in_record_date}
    group by 1,2,3
),

liveline_minute_concurrents AS (
    -- Add counts for all channels on the game/minute
    SELECT
        game_id,
        time_minute,
        COALESCE(SUM(ccu),0) AS game_ccu
    FROM liveline_channel
    GROUP BY 1,2
),

liveline_concurrents AS (
    SELECT
      game_id,
      AVG(game_ccu) AS avg_concurrents,
      MAX(game_ccu) AS max_concurrents
    FROM liveline_minute_concurrents
    GROUP BY 1
),

liveline_peak_time_ranks AS (
    SELECT game_id,
        time_minute,
        ROW_NUMBER() OVER(PARTITION BY game_id
            ORDER BY game_ccu DESC, time_minute ASC) AS views_minute_rank
    FROM liveline_minute_concurrents
),

liveline_peak_time AS (
    SELECT game_id,
        time_minute AS peak_time_minute
    FROM liveline_peak_time_ranks
    WHERE views_minute_rank = 1
),

create_clip AS (
    SELECT ga.id AS game_id,
        clip_id
    FROM tahoe."create_clip" cc
    INNER JOIN game_aliases ga ON CAST(LOWER(cc.game) AS VARCHAR(64)) = ga.alias
    WHERE {time_utc_in_record_date}
),

create_clip_agg AS (
    SELECT cc.game_id,
        COUNT(DISTINCT cc.clip_id) AS created
    FROM create_clip cc
    GROUP BY 1
),

clips AS (
    SELECT game_id,
        url
    FROM video_play vp
    WHERE (vod_type = 'clip' OR content_mode = 'clip')
        AND url IS NOT NULL
        AND url NOT LIKE 'https://www.twitch.tv/%%/manager/clips/channel'
        AND url != 'https://clips.twitch.tv/create'
),

clips_tops AS (
    SELECT ut.game_id,
        ut.url AS url,
        eut.url AS url_embed
    FROM (
        SELECT game_id, url, cnt
        FROM (
            SELECT game_id, url, COUNT(1) AS cnt,
                ROW_NUMBER() OVER(PARTITION BY game_id ORDER BY cnt DESC) AS cnt_rank
            FROM clips
            WHERE url NOT LIKE 'https://clips.twitch.tv/embed?clip=%%'
            GROUP BY 1, 2
        )
        WHERE cnt_rank = 1
    ) ut
    INNER JOIN (
        SELECT game_id, url, cnt
        FROM (
            SELECT game_id, url, COUNT(1) AS cnt,
                ROW_NUMBER() OVER(PARTITION BY game_id ORDER BY cnt DESC) AS cnt_rank
            FROM clips
            WHERE url LIKE 'https://clips.twitch.tv/embed?clip=%%'
            GROUP BY 1, 2
        )
        WHERE cnt_rank = 1
    ) eut
    ON ut.game_id = eut.game_id
),

extension_renderers AS (
    SELECT extension_id,
        channel_id,
        DATE_TRUNC('minute', time_utc) AS time_minute
    FROM tahoe."extension_render"
    WHERE {time_utc_in_record_date}
),

extension_by_game AS (
    SELECT mb.game_id,
        COUNT(DISTINCT er.channel_id) AS channel_count,
        COUNT(DISTINCT er.extension_id) AS extension_count
    FROM minute_broadcast mb
    INNER JOIN extension_renderers AS er
        ON er.channel_id = mb.channel_id AND er.time_minute = mb.time_minute
    GROUP BY 1
)

SELECT %(record_date)s::DATE                    AS "record_date",
    g.name                                      AS "game",
    g.id                                        AS "game_id",
    COALESCE(vp.live_views, 0)                  AS "live_views",
    COALESCE(vp.non_live_views, 0)              AS "non_live_views",
    COALESCE(vp.total_unique_viewers, 0)        AS "total_unique_viewers",
    COALESCE(vp.live_unique, 0)                 AS "live_uniques",
    COALESCE(vp.non_live_unique, 0)             AS "non_live_uniques",
    COALESCE(llc.avg_concurrents, 0)            AS "average_concurrent_viewers",
    COALESCE(llc.max_concurrents, 0)            AS "peak_concurrent_viewers",
    llcpk.peak_time_minute                      AS "peak_time_concurrent_viewers",
    COALESCE(hwl.hours_watched_live, 0)         AS "live_hours",
    COALESCE(hwnl.hours_watched_non_live, 0)    AS "non_live_hours",
    COALESCE(bc.uniques, 0)                     AS "unique_broadcasters",
    COALESCE(bc.hours_broadcasted, 0)           AS "hours_broadcast",
    COALESCE(bcc.avg_concurrents, 0)            AS "average_concurrent_broadcasters",
    COALESCE(bcc.max_concurrents, 0)            AS "peak_concurrent_broadcasters",
    bcpk.peak_time_minute                       AS "peak_time_concurrent_broadcasters",
    COALESCE(chat.live_unique_chat_participants, 0) AS "live_unique_chat_participants",
    COALESCE(chat.total_live_chat_messages_sent, 0) AS "total_live_chat_messages_sent",
    COALESCE(ext.channel_count, 0)              AS "unique_active_channels_with_extension",
    COALESCE(ext.extension_count, 0)            AS "unique_active_extensions",
    COALESCE(cca.created, 0)                    AS "clips_created",
    COALESCE(vp.clips_views, 0)                 AS "clip_views",
    ct.url                                      AS "top_clip_url",
    ct.url_embed                                AS "top_clip_url_embed"
FROM active_games g
LEFT JOIN views_aggr vp                      ON g.id = vp.game_id
LEFT JOIN hours_watched_live_aggr hwl        ON g.id = hwl.game_id
LEFT JOIN hours_watched_nonlive_aggr hwnl    ON g.id = hwnl.game_id
LEFT JOIN broadcasters bc                    ON g.id = bc.game_id
LEFT JOIN broadcasters_concurrents bcc       ON g.id = bcc.game_id
LEFT JOIN broadcasters_concurrents_peak bcpk ON g.id = bcpk.game_id
LEFT JOIN chat_live_aggregate chat           ON g.id = chat.game_id
LEFT JOIN liveline_concurrents llc           ON g.id = llc.game_id
LEFT JOIN liveline_peak_time llcpk           ON g.id = llcpk.game_id
LEFT JOIN create_clip_agg cca                ON g.id = cca.game_id
LEFT JOIN clips_tops ct                      ON g.id = ct.game_id
LEFT JOIN extension_by_game ext              ON g.id = ext.game_id
""".format(
    time_utc_in_record_date=time_utc_in_record_date)

# Rows to INSERT in Aurora from the SELECT query in Redshift.
FIELDS = [
    'record_date', 'game', 'game_id', 'live_views', 'non_live_views', 'total_unique_viewers',
    'live_uniques', 'non_live_uniques', 'average_concurrent_viewers',
    'peak_concurrent_viewers', 'peak_time_concurrent_viewers', 'live_hours', 'non_live_hours',
    'unique_broadcasters', 'hours_broadcast', 'average_concurrent_broadcasters',
    'peak_concurrent_broadcasters', 'peak_time_concurrent_broadcasters',
    'live_unique_chat_participants', 'total_live_chat_messages_sent',
    'unique_active_channels_with_extension',
    'unique_active_extensions', 'clips_created', 'clip_views', 'top_clip_url', 'top_clip_url_embed'
    ]

UPDATE_DOMAIN_REPORT_META_DATA = """
BEGIN;
DELETE FROM latest_domain_report_meta_data WHERE domain = 'games' AND report = 'overview_v2';
INSERT INTO latest_domain_report_meta_data
SELECT %(end_date)s
    , 'games'
    , 'overview_v2'
    , game_id
    , MAX(game)
    , MIN(record_date)
    , MAX(record_date)
FROM games_overview_v2
WHERE record_date >= %(start_date)s AND record_date < %(end_date)s
GROUP BY game_id
HAVING sum(hours_broadcast) > 5;
COMMIT;
"""
