# Load balancer for the airflow admin interface.
# Only accessible through WPA2
module "master_endpoint" {
  source            = "../loadbalancer"
  account           = "${var.account}"
  environment       = "${var.environment}"
  name              = "${var.service}"
  vpc               = "${var.vpc}"
  subnets           = ["${split(",", var.subnets_public)}"]
  security_groups   = ["${var.security_group_wpa2}"]
  internal          = false
  target_group_port = "8080"
  protocol          = "HTTP"
  certificate_arn   = "${var.airflow_certificate_arn}"
  region            = "${var.region}"

  healthy_threshold     = 2
  unhealthy_threshold   = 5
  health_check_timeout  = 10
  health_check_path     = "/static/bootstrap-toggle.min.css"
  health_check_interval = 50
  health_check_port     = "traffic-port"
  alert_arn             = ["${var.alert_arn}"]
}

resource "aws_cloudwatch_metric_alarm" "elb_unhealthy" {
  alarm_name          = "piper-airflow-elb-unhealthy"
  comparison_operator = "GreaterThanThreshold"
  evaluation_periods  = "3"
  metric_name         = "UnHealthyHostCount"
  namespace           = "AWS/ELB"
  period              = "180"
  statistic           = "Maximum"
  threshold           = "0"
  alarm_actions       = ["${var.alert_arn}"]
  ok_actions          = ["${var.alert_arn}"]
  alarm_description   = "Notify pagerduty if any hosts are unhealthy for more than 9 minutes."

  dimensions {
    LoadBalancerName = "${module.master_endpoint.alb_name}"
  }
}

resource "aws_cloudwatch_metric_alarm" "alb_5xx" {
  alarm_name          = "piper-airflow-alb-5xx"
  comparison_operator = "GreaterThanThreshold"
  evaluation_periods  = "3"
  metric_name         = "HTTPCode_ELB_5XX"
  namespace           = "AWS/ApplicationELB"
  period              = "120"
  statistic           = "Maximum"
  threshold           = "0"
  alarm_actions       = ["${var.alert_arn}"]
  ok_actions          = ["${var.alert_arn}"]
  alarm_description   = "Notify pagerduty if any hosts are unhealthy for more than 6 minutes."
  treat_missing_data  = "notBreaching"

  dimensions {
    LoadBalancerName = "${module.master_endpoint.alb_name}"
  }
}

# Load balancer for the airflow admin interface,
# Only accessible through Twitch-VPN
module "master_endpoint_twvpn" {
  source            = "../loadbalancer"
  account           = "${var.account}"
  environment       = "${var.environment}"
  name              = "${var.service}-twvpn"
  vpc               = "${var.vpc}"
  subnets           = ["${split(",", var.subnets)}"]
  security_groups   = ["${var.security_group}"]
  internal          = true
  target_group_port = "8080"
  protocol          = "HTTP"
  certificate_arn   = "${var.twvpn_certificate_arn}"
  region            = "${var.region}"

  healthy_threshold     = 2
  unhealthy_threshold   = 5
  health_check_timeout  = 10
  health_check_path     = "/static/bootstrap-toggle.min.css"
  health_check_interval = 50
  health_check_port     = "traffic-port"
  alert_arn             = ["${var.alert_arn}"]
}

# Load balancer for flower (Celery workers monitor).
# Only accessible through WPA2
module "master_flower_endpoint" {
  source            = "../loadbalancer"
  account           = "${var.account}"
  environment       = "${var.environment}"
  name              = "${var.service}-flower"
  vpc               = "${var.vpc}"
  subnets           = ["${split(",", var.subnets_public)}"]
  security_groups   = ["${var.security_group_wpa2}"]
  internal          = false
  target_group_port = "5555"
  protocol          = "HTTP"
  certificate_arn   = "${var.flower_certificate_arn}"
  region            = "${var.region}"

  healthy_threshold     = 2
  unhealthy_threshold   = 5
  health_check_timeout  = 10
  health_check_path     = "/static/bootstrap-toggle.min.css"
  health_check_interval = 50
  health_check_port     = "8080"

  alert_arn             = ["${var.alert_arn}"]
}

resource "aws_cloudwatch_metric_alarm" "alb_flower_unhealthy" {
  alarm_name          = "piper-airflow-flower-alb-unhealthy"
  comparison_operator = "GreaterThanThreshold"
  evaluation_periods  = "3"
  metric_name         = "UnHealthyHostCount"
  namespace           = "AWS/ApplicationELB"
  period              = "180"
  statistic           = "Maximum"
  threshold           = "0"
  alarm_actions       = ["${var.alert_arn}"]
  ok_actions          = ["${var.alert_arn}"]
  alarm_description   = "Notify pagerduty if any hosts are unhealthy for more than 9 minutes."

  dimensions {
    LoadBalancerName = "${module.master_flower_endpoint.alb_name}"
  }
}

resource "aws_cloudwatch_metric_alarm" "alb_flower_5xx" {
  alarm_name          = "piper-airflow-flower-alb-5xx"
  comparison_operator = "GreaterThanThreshold"
  evaluation_periods  = "3"
  metric_name         = "HTTPCode_ELB_5XX"
  namespace           = "AWS/ApplicationELB"
  period              = "120"
  statistic           = "Maximum"
  threshold           = "0"
  alarm_actions       = ["${var.alert_arn}"]
  ok_actions          = ["${var.alert_arn}"]
  alarm_description   = "Notify pagerduty if any hosts are unhealthy for more than 6 minutes."
  treat_missing_data  = "notBreaching"

  dimensions {
    LoadBalancerName = "${module.master_flower_endpoint.alb_name}"
  }
}

# Auto Scaling Group for the master instance,
# that is running airflow and flower (python webapps)
module "master_asg" {
  source                 = "../asg-instances-codedeploy"

  account                = "${var.account}"
  subnet_ids             = ["${split(",", var.subnets)}"]
  vpc_security_group_ids = ["${var.security_group}"]
  service                = "${var.service}"
  role                   = "airflow-master"
  environment            = "${var.environment}"
  ami                    = "${var.ami}"
  owner                  = "${var.owner}"
  region                 = "${var.region}"
  target_group_arns      = ["${module.master_endpoint.target_group_arn}", "${module.master_endpoint_twvpn.target_group_arn}", "${module.master_flower_endpoint.target_group_arn}"]
  sandstorm_assume_arn   = "${var.sandstorm_assume_arn}"
  min_elb_capacity       = 1
  min_count              = 0
  max_count              = 1
  desired_capacity       = 1
  instance_type          = "${var.master_type}"
  root_block_device_size = "512"
}

resource "aws_cloudwatch_metric_alarm" "master_asg_failed" {
  alarm_name          = "piper-airflow-master-asg-failed"
  comparison_operator = "GreaterThanThreshold"
  evaluation_periods  = "3"
  metric_name         = "StatusCheckFailed"
  namespace           = "AWS/EC2"
  period              = "120"
  statistic           = "Maximum"
  threshold           = "0"
  alarm_actions       = ["${var.alert_arn}"]
  ok_actions          = ["${var.alert_arn}"]
  alarm_description   = "Notify pagerduty if any hosts are unhealthy for more than 6 minutes."

  dimensions {
    AutoScalingGroupName = "${module.master_asg.asg_name}"
  }
}

resource "aws_cloudwatch_metric_alarm" "master_asg_cpu" {
  alarm_name          = "piper-airflow-master-asg-cpu"
  comparison_operator = "GreaterThanThreshold"
  evaluation_periods  = "3"
  metric_name         = "CPUUtilization"
  namespace           = "AWS/EC2"
  period              = "180"
  statistic           = "Maximum"
  threshold           = "75"
  alarm_actions       = ["${var.alert_arn}"]
  ok_actions          = ["${var.alert_arn}"]
  alarm_description   = "Notify pagerduty if any have more tahn 75% CPU for more than 9 minutes."

  dimensions {
    AutoScalingGroupName = "${module.master_asg.asg_name}"
  }
}

output "master_iam_role_arn" {
  value = "${module.master_asg.asg_iam_role_arn}"
}
