module "worker_asg" {
  source                  = "../asg-instances-codedeploy"

  account                 = "${var.account}"
  subnet_ids              = ["${split(",", var.subnets)}"]
  vpc_security_group_ids  = ["${var.security_group}"]
  service                 = "${var.service}"
  role                    = "airflow-worker"
  environment             = "${var.environment}"
  ami                     = "${var.ami}"
  owner                   = "${var.owner}"
  sandstorm_assume_arn    = "${var.sandstorm_assume_arn}"
  target_group_arns       = []
  min_elb_capacity        = 0
  min_count               = "${var.min_workers}"
  max_count               = "${var.max_workers}"
  desired_capacity        = "${var.desired_workers}"

  instance_type           = "${var.worker_type}"
  root_block_device_size  = "256"
}

resource "aws_cloudwatch_metric_alarm" "worker_asg_failed" {
    alarm_name = "piper-airflow-worker-asg-failed"
    comparison_operator = "GreaterThanThreshold"
    evaluation_periods = "3"
    metric_name = "StatusCheckFailed"
    namespace = "AWS/EC2"
    period = "120"
    statistic = "Maximum"
    threshold = "0"
    alarm_actions = ["${var.alert_arn}"]
    ok_actions = ["${var.alert_arn}"]
    alarm_description = "Notify pagerduty if any hosts are unhealthy for more than 6 minutes."

    dimensions {
        AutoScalingGroupName = "${module.worker_asg.asg_name}"
    }
}

resource "aws_cloudwatch_metric_alarm" "worker_asg_cpu" {
    alarm_name = "piper-airflow-worker-asg-cpu"
    comparison_operator = "GreaterThanThreshold"
    evaluation_periods = "3"
    metric_name = "CPUUtilization"
    namespace = "AWS/EC2"
    period = "60"
    statistic = "Maximum"
    threshold = "90"
    alarm_actions = ["${var.alert_arn}"]
    ok_actions = ["${var.alert_arn}"]
    alarm_description = "Notify pagerduty if any have more than 90% CPU for more than 3 minutes."

    dimensions {
        AutoScalingGroupName = "${module.worker_asg.asg_name}"
    }
}

output "worker_iam_role_arn" {
  value = "${module.worker_asg.asg_iam_role_arn}"
}

# IAM policy for S3

data "aws_iam_policy_document" "s3_access_policy_doc" {
    statement {
        actions = ["s3:*"]
        resources = ["*"]
    }
}

resource "aws_iam_policy" "s3_access_policy" {
    name = "${var.service}-airflow-worker-${var.environment}-s3-access"
    policy = "${data.aws_iam_policy_document.s3_access_policy_doc.json}"
}

resource "aws_iam_role_policy_attachment" "s3_access_policy_attachment" {
    role = "${module.worker_asg.asg_iam_role}"
    policy_arn = "${aws_iam_policy.s3_access_policy.arn}"
}

# IAM policy for CloudWatch

data "aws_iam_policy_document" "cloudwatch_policy_doc" {
    statement {
        actions = ["logs:CreateLogGroup", "logs:CreateLogStream", "logs:PutLogEvents", "logs:DescribeLogStreams", "ec2:DescribeTags"]
        resources = ["arn:aws:logs:*:*:*"]
    }
}

resource "aws_iam_policy" "cloudwatch_policy" {
    name = "${var.service}-airflow-worker-${var.environment}-cloudwatch"
    policy = "${data.aws_iam_policy_document.cloudwatch_policy_doc.json}"
}

resource "aws_iam_role_policy_attachment" "cloudwatch_policy_attachment" {
    role = "${module.worker_asg.asg_iam_role}"
    policy_arn = "${aws_iam_policy.cloudwatch_policy.arn}"
}
