# Simple ASG Spinup + CodeDeploy Agent
# If you're using this, make sure you read and understand the codedeploy spinup instructions in https://git-aws.internal.justin.tv/d8a/codedeploy-app
# Copied from: https://git-aws.internal.justin.tv/d8a/asg-instances-codedeploy

variable "template_name" {
    default = "user_data.tpl"
}

# Basic IAM Role
resource "aws_iam_role" "iam_role" {
    name = "${var.service}-${var.role}-${var.environment}"
    path = "/"
    assume_role_policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Action": "sts:AssumeRole",
            "Principal": {
                "Service": "ec2.amazonaws.com"
            },
            "Effect": "Allow",
            "Sid": ""
        }
    ]
}
EOF
}

data "aws_iam_policy_document" "sandstorm_assume_policy_doc" {
    statement {
        actions = ["sts:AssumeRole"]
        resources = ["${var.sandstorm_assume_arn}"]
    }
}

resource "aws_iam_policy" "sandstorm_assume_policy" {
    name = "${var.service}-${var.role}-${var.environment}-sandstorm"
    policy = "${data.aws_iam_policy_document.sandstorm_assume_policy_doc.json}"
}

resource "aws_iam_role_policy_attachment" "sandstorm_assume_policy_attachment" {
    role = "${aws_iam_role.iam_role.name}"
    policy_arn = "${aws_iam_policy.sandstorm_assume_policy.arn}"
}

data "aws_iam_policy" "AmazonEC2RoleforSSM" {
  arn = "arn:aws:iam::aws:policy/service-role/AmazonEC2RoleforSSM"
}

resource "aws_iam_role_policy_attachment" "ssm_attachment" {
  role = "${aws_iam_role.iam_role.name}"
  policy_arn = "${data.aws_iam_policy.AmazonEC2RoleforSSM.arn}"
}

resource "aws_iam_instance_profile" "instance_profile" {
    name = "${var.service}-${var.role}-${var.environment}"
    role = "${aws_iam_role.iam_role.id}"
    depends_on = ["aws_iam_role.iam_role"]
}

data "template_file" "user_data" {
    template = "${file(join("/", list(path.module, var.template_name)))}"
    vars {
        role = "${var.role}"
        cluster = "${var.cluster == "" ? var.role : var.cluster }"
        env = "${lookup(var.environment_tagname, var.environment)}"
        owner = "${var.owner}"
        environment = "${var.environment}"
        puppet = "${lookup(var.puppet_server, var.puppet)}"
        dns_region = "${lookup(var.dns_region, var.region)}"
        region = "${var.region}"
        sandstorm_role = "${var.sandstorm_assume_arn}"
    }
}

resource "aws_launch_configuration" "asg_launch" {

    image_id = "${var.ami}"
    associate_public_ip_address = "false"
    instance_type = "${var.instance_type}"
    security_groups = "${var.vpc_security_group_ids}"
    iam_instance_profile = "${aws_iam_instance_profile.instance_profile.id}"

    root_block_device {
        volume_size = "${var.root_block_device_size}"
    }

    lifecycle {
        create_before_destroy = true
    }
    
    user_data = "${data.template_file.user_data.rendered}"
}

resource "aws_autoscaling_group" "asg" {

    name = "${var.service}-${var.role}-${var.environment}"
    launch_configuration = "${aws_launch_configuration.asg_launch.name}"

    max_size = "${var.max_count}"
    min_size = "${var.min_count}"
    desired_capacity = "${var.desired_capacity}"
    vpc_zone_identifier = "${var.subnet_ids}"
    target_group_arns = ["${var.target_group_arns}"]
    min_elb_capacity = "${var.min_elb_capacity}"

    tag {
        key = "Name"
        value = "${var.role}-${var.environment}"
        propagate_at_launch = true
    }

    tag {
        key = "Environment"
        value = "${lookup(var.environment_tagname, var.environment)}"
        propagate_at_launch = true
    }

    tag {
        key = "Owner"
        value = "${var.owner}@twitch.tv"
        propagate_at_launch = true
    }

    tag {
        key = "Service"
        value = "${var.owner}/${var.service}/${var.role}"
        propagate_at_launch = true
    }

    tag {
        key = "DeployGroup"
        value = "${var.owner}/${var.service}/${var.deploy_group}"
        propagate_at_launch = true
    }
}
