variable "environment" {}
variable "project_tag" {}
variable "owner_tag" {}
variable "subnet_ids" {}
variable "vpc_group_id" {}
variable "master_password" {}
variable "database" {}
variable "node_count" {}
variable "alert_arn" {}
variable "public_ip" {
  default = ""
}

variable "account_id" {}
variable "region" {}
variable "airlow_master_role_arn" {}
variable "airlow_worker_role_arn" {}
variable "bucket" {}

variable "roles" {
  type    = "list"
  default = []
}

resource "aws_redshift_parameter_group" "redshift_params" {
  name   = "piper-${var.database}-reports-params"
  family = "redshift-1.0"

  parameter {
    name = "wlm_json_configuration"

    # https://docs.aws.amazon.com/redshift/latest/mgmt/workload-mgmt-config.html
    # Two workload pools- first is a 1-concurrency pool for query group 'maintenance', other is a 5-concurrency default pool for all other queries
    # Default pool & short query values are the redshift default
    value = "[{\"query_concurrency\":1,\"max_execution_time\":0,\"query_group\":[\"maintenance\"],\"query_group_wild_card\":0,\"user_group\":[],\"user_group_wild_card\":0},{\"query_concurrency\":5,\"max_execution_time\":0,\"query_group\":[],\"query_group_wild_card\":0,\"user_group\":[],\"user_group_wild_card\":0},{\"short_query_queue\":true,\"max_execution_time\":5000}]"
  }

  parameter {
    name  = "enable_user_activity_logging"
    value = "true"
  }
}

resource "aws_redshift_subnet_group" "redshift_subnet" {
  name       = "piper-${var.database}-redshift-subnet"
  subnet_ids = ["${split(",", var.subnet_ids)}"]

  tags {
    Project     = "${var.project_tag}"
    Owner       = "${var.owner_tag}"
    Environment = "${var.environment}"
  }
}

resource "aws_iam_role" "redshift_copy_unload" {
  name = "piper-${var.database}-redshift-copy-unload"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Principal": {
        "Service": "redshift.amazonaws.com"
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF
}

resource "aws_redshift_cluster" "redshift" {
  cluster_identifier           = "piper-${var.database}-redshift"
  database_name                = "${var.database}"
  master_username              = "piper"
  master_password              = "${var.master_password}"
  node_type                    = "dc2.8xlarge"
  cluster_type                 = "${var.node_count > 1 ? "multi-node" : "single-node"}"
  number_of_nodes              = "${var.node_count}"
  vpc_security_group_ids       = ["${var.vpc_group_id}"]
  cluster_subnet_group_name    = "${aws_redshift_subnet_group.redshift_subnet.name}"
  cluster_parameter_group_name = "${aws_redshift_parameter_group.redshift_params.name}"

  cluster_version       = "1.0"
  allow_version_upgrade = "true"
  publicly_accessible   = "true"
  encrypted             = "false"
  enhanced_vpc_routing  = "false"
  elastic_ip            = "${var.public_ip}"
  iam_roles             = ["${aws_iam_role.redshift_copy_unload.arn}", "${var.roles}"]

  preferred_maintenance_window = "wed:21:15-wed:21:45"
  skip_final_snapshot   = "true"

  logging {
    enable      = "true"
    bucket_name = "science-tahoe-read-replica-logs"
  }
}

resource "aws_cloudwatch_metric_alarm" "redshift_cpu" {
  alarm_name          = "piper-${var.database}-redshift-cpu"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  evaluation_periods  = "9"
  metric_name         = "CPUUtilization"
  namespace           = "AWS/Redshift"
  period              = "600"
  statistic           = "Maximum"
  threshold           = "90"
  alarm_actions       = ["${var.alert_arn}"]
  ok_actions          = ["${var.alert_arn}"]
  alarm_description   = "Notify pagerduty if CPU sustains above 90% for 90 minutes."

  dimensions {
    ClusterIdentifier = "${aws_redshift_cluster.redshift.cluster_identifier}"
  }
}

resource "aws_cloudwatch_metric_alarm" "redshift_health" {
  alarm_name          = "piper-${var.database}-redshift-health"
  comparison_operator = "LessThanThreshold"
  evaluation_periods  = "5"
  metric_name         = "HealthStatus"
  namespace           = "AWS/Redshift"
  period              = "300"
  statistic           = "Minimum"
  threshold           = "1"
  alarm_actions       = ["${var.alert_arn}"]
  ok_actions          = ["${var.alert_arn}"]
  alarm_description   = "Notify pagerduty if health is below 1 for 25 minutes"
  datapoints_to_alarm = "5"

  dimensions {
    ClusterIdentifier = "${aws_redshift_cluster.redshift.cluster_identifier}"
  }
}

output "cluster_host" {
  value = "${aws_redshift_cluster.redshift.endpoint}"
}

output "cluster_database" {
  value = "${aws_redshift_cluster.redshift.database_name}"
}

data "aws_iam_policy_document" "redshift_access_policy_doc" {
  statement {
    actions = [
      "redshift:DescribeClusters",
    ]

    resources = ["*"]
  }

  statement {
    actions = [
      "redshift:ModifyClusterIamRoles",
      "redshift:CreateCluster",
    ]

    resources = [
      "arn:aws:redshift:${var.region}:${var.account_id}:${aws_redshift_cluster.redshift.cluster_identifier}",
    ]
  }

  statement {
    actions = [
      "iam:PassRole",
    ]

    resources = [
      "${var.airlow_master_role_arn}",
      "${var.airlow_worker_role_arn}",
    ]
  }

  statement {
    actions = [
      "s3:Get*",
      "s3:List*",
      "s3:Put*",
      "s3:Delete*",
      "s3:CreateBucket",
    ]

    resources = [
      "arn:aws:s3:::${var.bucket}",
      "arn:aws:s3:::${var.bucket}/*",
    ]
  }
}

resource "aws_iam_policy" "redshift_access_policy" {
  name   = "${aws_iam_role.redshift_copy_unload.name}-policy"
  policy = "${data.aws_iam_policy_document.redshift_access_policy_doc.json}"
}

resource "aws_iam_role_policy_attachment" "redshift_access_policy_attachment" {
  role       = "${aws_iam_role.redshift_copy_unload.name}"
  policy_arn = "${aws_iam_policy.redshift_access_policy.arn}"
}
