terraform {
  backend "s3" {
    bucket = "twitch-salesforce-pipeline-terraform-prod"
    key    = "terraform"
    region = "us-west-2"
  }
}

provider "aws" {
  region = "us-west-2"
}

variable "boomi_user" {
  description = "user in another account that is used by boomi for reading bucket"
  default     = "arn:aws:iam::868653755646:user/service_boomi"
}

resource "aws_kms_key" "payments" {
  description             = "This key is used to encrypt payments bucket objects"
  deletion_window_in_days = 10
}

resource "aws_s3_bucket" "payments" {
  bucket = "twitch-salesforce-payments-prod"
  acl    = "private"

  logging = {
    target_bucket = "${aws_s3_bucket.payments_logs.id}"
    target_prefix = "log/"
  }

  server_side_encryption_configuration {
    rule {
      apply_server_side_encryption_by_default {
        kms_master_key_id = "${aws_kms_key.payments.arn}"
        sse_algorithm     = "aws:kms"
      }
    }
  }
}

resource "aws_s3_bucket" "payments_logs" {
  bucket = "twitch-salesforce-payments-prod-logs"
  acl    = "log-delivery-write"
}

data "aws_iam_policy_document" "payments_bucket" {
  statement {
    sid    = "list permission"
    effect = "Allow"

    actions = [
      "s3:ListBucket",
    ]

    principals = {
      type = "AWS"

      identifiers = [
        "${aws_iam_role.payments_writer.arn}",
        "${var.boomi_user}",
      ]
    }

    resources = [
      "${aws_s3_bucket.payments.arn}",
    ]
  }

  statement {
    sid    = "read permission"
    effect = "Allow"

    actions = [
      "s3:GetObject",
      "s3:DeleteObject",
    ]

    principals = {
      type = "AWS"

      identifiers = [
        "${aws_iam_role.payments_writer.arn}",
        "${var.boomi_user}",
      ]
    }

    resources = [
      "${aws_s3_bucket.payments.arn}/*",
    ]
  }

  statement {
    sid    = "write permission"
    effect = "Allow"

    actions = [
      "s3:PutObject",
    ]

    principals = {
      type = "AWS"

      identifiers = [
        "${aws_iam_role.payments_writer.arn}",
      ]
    }

    resources = [
      "${aws_s3_bucket.payments.arn}/*",
    ]
  }
}

resource "aws_s3_bucket_policy" "payments_bucket" {
  bucket = "${aws_s3_bucket.payments.bucket}"
  policy = "${data.aws_iam_policy_document.payments_bucket.json}"
}

output "payment_bucket_arn" {
  value = "${aws_s3_bucket.payments.arn}"
}
