import cdk = require('@aws-cdk/core');
import autoscaling = require('@aws-cdk/aws-autoscaling');
import ec2 = require('@aws-cdk/aws-ec2');
import ecs = require('@aws-cdk/aws-ecs');
import ecs_patterns = require('@aws-cdk/aws-ecs-patterns');
import elb = require('@aws-cdk/aws-elasticloadbalancingv2');
import iam = require('@aws-cdk/aws-iam');
import path = require('path'); // yarn add --dev @types/node

export class StatsdTransformerStack extends cdk.Stack {
  constructor(scope: cdk.Construct, id: string, props?: cdk.StackProps) {
    super(scope, id, props);

    const statsd_port = 8125;

    // FIXME: allow plugging in an existing vpc
    const vpc = new ec2.Vpc(this, 'StatsdXFrm_VPC', { maxAzs: 2 });

    const asg = new autoscaling.AutoScalingGroup(this, 'StatsdXFrm_ASG', {
      instanceType: ec2.InstanceType.of(ec2.InstanceClass.C5D, ec2.InstanceSize.LARGE),
      updateType: autoscaling.UpdateType.REPLACING_UPDATE,
      machineImage: new ecs.EcsOptimizedAmi(),
      desiredCapacity: 1,
      vpc,
    });

    // security groups for asg
    // statsd <3s UDP, so allow _everything_
    asg.connections.allowFrom(ec2.Peer.ipv4(vpc.vpcCidrBlock), ec2.Port.allTraffic());

    // policy, asg user data for ssm (so we always update that)
    asg.addToRolePolicy(new iam.PolicyStatement({
      actions: [
        'ec2messages:*',
        'ssm:UpdateInstanceInformation',
        'ssmmessages:*',
      ],
      resources: [ '*' ],
    }));
    asg.addUserData('yum -y install https://s3.amazonaws.com/ec2-downloads-windows/SSMAgent/latest/linux_amd64/amazon-ssm-agent.rpm');

    // glue to cluster
    const cluster = new ecs.Cluster(this, 'StatsdXFrm_ECSCL', { vpc });
    cluster.addAutoScalingGroup(asg);

    // incoming traffic LB - using statsd port but TCP/UDP so we have health checks!
    const lb = new elb.NetworkLoadBalancer(this, 'StatsdXFrm_LB', {
      vpc: vpc,
      internetFacing: false
    });
    const listener = lb.addListener('StatsdXFrm_LBL', {
      port: statsd_port,
      protocol: elb.Protocol.TCP_UDP
    });

    // this part is *magic* - take local files and turn it into an ECS task!
    const ecs_service = new ecs_patterns.NetworkLoadBalancedEc2Service(this, 'StatsdXFrm_SVC', {
      cluster,
      loadBalancer: lb,
      memoryLimitMiB: 512,
      taskImageOptions: {
        image: ecs.ContainerImage.fromAsset(path.join(__dirname, '..', 'telegraf-task')),
        containerPort: statsd_port
      }
    });
    const targets = listener.addTargets('StatsdXFrm_SVC', {
      port: statsd_port,
      targets: [ecs_service.service],
    });
    // I hate the docs for doing UDP in this thing
    const cfnLsnTarget = targets.node.findChild('Resource') as elb.CfnTargetGroup;
    cfnLsnTarget.addPropertyOverride('Protocol', 'TCP_UDP');

  } //constructor
}
