define twitch_bare_metal::bond_setup::interface {
  include twitch_bare_metal::bond_setup

  # here we source our config hash, namespaced on current iface name
  $bond_config = $twitch_bare_metal::bond_setup::bond_config_merged[$name]

  # if bond_config doesnt have interface name,
  # like with slaves, we *must* have a hash
  if $bond_config {
    validate_hash($bond_config)
    # lets see if this bond device should be absent
    $ensure = pick($bond_config['ensure'], 'present')
  } else {
    # if we dont have a bond_config for this interface
    # we want it present, as its a slave interface (ethX)
    $ensure = 'present'
  }

  # dont waste time if we dont want it
  if ($ensure != 'absent') {
    case $name {
      # bond alias
      /^bond\d+\:\d+$/: {
        $method  = 'static'
        # ipaddress and netmask need to be set in bond_config
        # otherwise we have no clue what to set it to
        if !($bond_config['address'] and $bond_config['netmask']) {
          fail("need 'address' and 'netmask' for alias $name")
        } else {
          # validate we have actual IP addresses
          if !(is_ip_address($bond_config['address'])) {
            fail("address is not an ip (${bond_config['address']})")
          }
          if !(is_ip_address($bond_config['netmask'])) {
            fail("netmask is not an ip (${bond_config['netmask']})")
          }
        }
        $changes = [
          "set iface[. = '${name}']/address ${bond_config['address']}",
          "set iface[. = '${name}']/netmask ${bond_config['netmask']}",
        ]
      }
      # bond interface
      /^bond\d+$/: {
        # bond mode is global to the bond
        $bond_mode = $twitch_bare_metal::bond_setup::bond_mode
        $method    = 'dhcp'
        # master interface needs address/netmask/gateway + bond_mode
        case $bond_mode {
          'ad': {
            $mode = [
              "set iface[. = '${name}']/bond-mode 802.3ad",
              "set iface[. = '${name}']/bond-lacp-rate 1",
            ]
          }
          'rr': {
            $mode = [
              "set iface[. = '${name}']/bond-mode balance-rr",
            ]
          }
          default: {
            # we don't use any other modes
            fail("\$bond_mode is unknown (${bond_mode})")
          }
        }
        $changes       = concat($mode, [
          "set iface[. = '${name}']/hwaddress 'ether ${twitch_bare_metal::bond_setup::macaddress_primary}'",
          "set iface[. = '${name}']/bond-slaves/1 none",
          "set iface[. = '${name}']/bond-miimon 100",
          "set iface[. = '${name}']/bond-xmit_hash_policy layer3+4",
        ])
      }
      # the slave interfaces
      /^eth\d+$/: {
        $method  = 'manual'
        $changes = [
          "set iface[. = '${name}']/bond-master bond0",
          "set iface[. = '${name}']/hardware-dma-ring-rx 2048",
          "set iface[. = '${name}']/hardware-irq-coalesce-rx-usecs 1",
        ]
      }
      default: {
        # we got something other than eth or bond
        fail("Unknown inteface for twitch_bare_metal::bond_setup $name")
      }
    }
    # all interfaces share common header
    # uses method from above
    $header = [
      "set auto[child::1 = '${name}']/1 ${name}",
      "set iface[. = '${name}'] ${name}",
      "set iface[. = '${name}']/family inet",
      "set iface[. = '${name}']/method ${method}",
    ]
    # entire array of file changes
    $file_changes = concat($header, $changes)
  }

  # we need to tell augeas to check if interface already exists in the file
  # so we dont clobber existing bonds that were maybe hand made
  # set $check_interface to the name of the bond interfaces we're trying
  # to configure.
  # says "change file ONLYIF I dont have any bond interface already"
  $check_interface = $name ? {
    /^bond\d+.*$/ => $name,
    default       => 'bond0'
  }
  interfaces::interface { $name:
    changes => $file_changes,
    ensure  => $ensure,
    onlyif  => "match iface[. = '${check_interface}'] size == 0"
  }
}
