#!/bin/bash

####################

# create_array.sh
# creates a raid0-backed xfs volume across the given devices
# usage: sudo bash create_array.sh /dev/sda /dev/sdb ...

# only supports xfs filesystems at the moment, and several options (raid level,
# filesystem label, and filesystem mount point) are currently hardcoded

# hardcoded options
raid_level=0
filesystem_label="cassandra"
filesystem_mount_point="/var/lib/cassandra/data"

####################

# bail out if we aren't root
if [ $(id -u) != "0" ]; then
    echo "This script must be executed by the root user. Exiting."
    exit 1
fi

# store all arguments (block device names) in one variable
block_devices="$*"
num_block_devices=$(echo $block_devices | wc -w)

remove_superblock () {
    # removes superblocks from block devices
    device=$1
    mdadm --zero-superblock $device
}

find_unused_volume() {
    # finds the first unused block device starting with /dev/md
    for n in $(seq 0 255); do
        target_device="/dev/md${n}"
        if [ ! -b $target_device ]; then
            echo $target_device
        return
        fi
    done
}

# saves this variable globally for use in other functions
raid_volume=$(find_unused_volume)

create_raid_volume () {
    # creates raid volume
    block_devices="$*"
    mdadm --create $raid_volume --level=$raid_level --raid-devices=$num_block_devices $block_devices
}

save_raid_configuration () {
    mdadm_conf="/etc/mdadm/mdadm.conf"
    echo "CREATE owner=root group=disk mode=0660 auto=yes" > $mdadm_conf
    echo "HOMEHOST <system>" >> $mdadm_conf
    echo "MAILADDR root" >> $mdadm_conf
    mdadm --detail --scan >> $mdadm_conf
}

create_xfs_filesystem () {
    # sets the size of the log section to 128M
    # sets the stripe unit to 64K and the stripe width to 256K
    # sets the number of allocation groups to 64
    mkfs.xfs \
        -l size=128m \
        -d su=64k,sw=4,agcount=64 \
        -L $filesystem_label \
        $raid_volume
}

mount_filesystem () {
    mount -t xfs $raid_volume $filesystem_mount_point
}

get_raid_volume_uuid () {
    raid_volume=$1
    echo $(blkid -s UUID | grep $raid_volume | cut -d\" -f2)
}

update_fstab () {
    # if $raid_volume is not in /etc/fstab, put it there
    raid_uuid=$(get_raid_volume_uuid $raid_volume)
    entry="UUID=$raid_uuid $filesystem_mount_point xfs noatime,nobarrier,logbufs=8 0 1"
    grep -q $raid_uuid /etc/fstab || {
        echo $entry >> /etc/fstab
    }
}

main () {
    # put it all together
    for device in $block_devices; do
        remove_superblock $device
    done
    create_raid_volume $block_devices
    save_raid_configuration
    create_xfs_filesystem $raid_volume
    mkdir -p $filesystem_mount_point
    mount_filesystem
    update_fstab
}

main
